/*********************************************************
 **
 ** FILE:    direc.c
 **		Directory operations
 ** CREATED: 21 SEP 92  Claus Brenner
 ** CHANGES: 
 **
 *********************************************************/

#include        <stdio.h>
#include	<sys/types.h>
#include	<sys/stat.h>
#include	<dirent.h>

#include	"misc.h"
#include	"direc.h"


/*--------------------------------------------------------
 ** FUNCTION: SetPathname
 ** PURPOSE:  Set the 'Pathname' variable in the given
 **		DirContents structure, check pathname
 **		first
 ** INPUTS:   DC - ptr to DirContents struct
 **		Pathname - new path name
 ** RETURNS:  TRUE - iff the new path is set
 **------------------------------------------------------*/

BOOLEAN SetPathname( DC, Pathname )
     struct DirContents *DC;
     char *Pathname;
     
{
  DIR	*tempdir;
  int	i;
  char	path[MAXPATHLEN];
  
  /* the path ends always with a '/' character */
  strcpy( path, Pathname );
  if( i = strlen(path) )
    {
      if( path[i-1] != '/' )
	strcat(path, "/");
    }
  else
    strcpy(path, "/");
  
  /* set new Pathname if it can be opened */
  if( tempdir = opendir(path) )
    {
      strcpy( DC->Pathname, path );
      closedir( tempdir );
      return(TRUE);
    }
  
  return(FALSE);
}


/*--------------------------------------------------------
 ** FUNCTION: AddPathname
 ** PURPOSE:  Add a string to the current path name
 ** INPUTS:   DC - ptr to DirContents struct
 **		AddPath - Path to be added, with no leading
 **		'/'-characters.
 ** RETURNS:  TRUE - iff AddPath was added.
 **------------------------------------------------------*/

BOOLEAN AddPathname( DC, AddPath )
     struct DirContents *DC;
     char *AddPath;
     
{
  char	path[MAXPATHLEN];
  
  strcpy(path, DC->Pathname);
  strcat(path, AddPath);
  
  return(SetPathname(DC, path));
}


/*--------------------------------------------------------
 ** FUNCTION: TruncPathname
 ** PURPOSE:  Go back from the current path to previous
 **		path 'levels'. E.g. TruncPathname 2 on
 **		/usr/local/fred will truncate everything
 **		to the right of the second '/' character,
 **		leaving /usr/ as current path.
 ** INPUTS:   DC - ptr to DirContents struct
 **		depth - no of '/' characters that shall
 **		remain in the path (must be greater than 0).
 ** RETURNS:  none (DC's Pathname is changed)
 **------------------------------------------------------*/

void TruncPathname( DC, depth )
     struct DirContents *DC;
     int depth;
     
{
  register char *p;
  
  /* to be safe */
  if( !depth ) return;
  
  for( p = DC->Pathname; *p != '\0'; ++p )
    if( *p == '/' )
      if( ! --depth ) break;
  
  /* if *p != '\0', p points to the right '/' character */
  if( *p != '\0' )
    *++p = '\0';
}



/*--------------------------------------------------------
 ** FUNCTION: SwapDirEntries
 ** PURPOSE:  swap two name pointers in a DirContents struct
 ** INPUTS:   DC - DirContents ptr
 **		First, Second - numbers of the elements to
 **		be swapped.
 ** RETURNS:  none.
 **------------------------------------------------------*/

void SwapDirEntries( DC, First, Second )
     struct DirContents *DC;
     int First, Second;
     
{
  char *temp;
  
  temp            = DC->Name[First];
  DC->Name[First] = DC->Name[Second];
  DC->Name[Second]= temp;
}


/*--------------------------------------------------------
 ** FUNCTION: QSDir
 ** PURPOSE:  Quicksort on a DirContents struct
 **		Note the remark in direc.h regarding
 **		the leading character in names.
 ** INPUTS:   DC - ptr to DirContents struct
 **		From, To - limits of the file to be sorted.
 ** RETURNS:  none (sorted entries in DC).
 **------------------------------------------------------*/

void QSDir( DC, From, To )
     struct DirContents *DC;
     int From, To;
     
{
  char	*pivot;
  int	smaller, i;
  
  if( From >= To )
    return;
  
  smaller = From;
  pivot = DC->Name[From]+1;
  
  for( i=From+1; i<=To; ++i )
    if( strcmp(DC->Name[i]+1, pivot) < 0 )
      SwapDirEntries( DC, ++smaller, i );
  
  SwapDirEntries( DC, From, smaller );
  QSDir( DC, From, smaller-1 );
  QSDir( DC, smaller+1, To );
}


/*--------------------------------------------------------
 ** FUNCTION: ScanDir
 ** PURPOSE:  Given the Pathname field of the DirContents
 **		struct, read in all Entries.
 ** INPUTS:   DC - ptr to DirContents struct
 ** RETURNS:  none (DC ist modified).
 **------------------------------------------------------*/

void ScanDir( DC )
     struct DirContents *DC;
     
{
  DIR	*dir;
  struct dirent *direntry;
  struct stat statbuffer;
  char	fullname[MAXPATHLEN];
  int	i;
  int	totalEntries = 0, dirEntries = 0;
  register char	*p = DC->EntryNames;	/* points to next free EntryNames character */
  register int	pcount = 0;		/* counts the characters in EntryNames so far */
  
  
  if( ! (dir = opendir(DC->Pathname)) )
    {
      /*  normally, this is never the case, since the above
       ** routines guarantee a Pathname that can be opened */
      fprintf(stderr,"Can't open the current directory");
      return;
    }
  
  for( direntry = readdir(dir); direntry != NULL; direntry = readdir(dir) )
    if( direntry->d_name[0] != '.' )
      {
	/* Try to copy the name to EntryNames, leave first character */
	if( pcount + (i=strlen(direntry->d_name)) + 1 <  DI_MAX_ENTRYSPACE )
	  {
	    strcpy(p+1, direntry->d_name);
	    pcount += i+2;
	  }
	else
	  break;
	
	/* Try to make a new Name entry */
	if( totalEntries < DI_MAX_ENTRIES )
	  DC->Name[totalEntries] = p;
	else
	  break;
	
	/* check if it is a file or a directory */
	strcpy(fullname, DC->Pathname);
	strcat(fullname, p+1);
	
	/* by default, the name belongs to a file */
	*p = ' ';
	if( !stat(fullname, &statbuffer) )
	  if( S_ISDIR(statbuffer.st_mode) )
	    {
	      SwapDirEntries( DC, totalEntries, dirEntries++ );
	      *p = '/';
	    }
	
	p += i+2;
	totalEntries++;
      }
  
  DC->NoOfEntries = totalEntries;
  DC->NoOfDirEntries = dirEntries;
  closedir(dir);
  
  /* Sort directories and files */
  QSDir( DC, 0, dirEntries-1 );
  QSDir( DC, dirEntries, totalEntries-1 );
}


/*--------------------------------------------------------
 ** FUNCTION: IsAFile
 ** PURPOSE:  Check for a number, if the corresponding
 **		entry in the DirContents structure is a file
 **		or a directory.
 ** INPUTS:   DC - ptr to a DirContents struct
 **		Number - entry number to be checked
 **		(entry numbering starts from 0)
 ** RETURNS:  TRUE - entry is a file or entry no is too large
 **		FALSE - entry is a directory
 **------------------------------------------------------*/

BOOLEAN IsAFile( DC, Number )
     struct DirContents *DC;
     int Number;
     
{
  if( Number >= DC->NoOfDirEntries )
    return(TRUE);
  else
    return(FALSE);
}


