/***************************************************
 ** PGUI - Parallaxis Graphical User Interface    **
 ** --------------------------------------------- **
 ** Studienarbeit Nr. 1201                        **
 ** "Erstellung einer graphischen Oberflaeche     **
 **  fuer das Parallaxis-System"                  **
 ** ============================================= **
 ** Revision    : 1.0                             **
 ** Author      : Stefan Frank                    **
 ** Last change : 16.11.93                        **
 ** Comments    :                                 **
 **  (1) creation of five form widgets            **
 **  (2) creation of command widgets              **
 **  (3) adaptation of the file select window     **
 **      (originally created by Claus Brenner)    **
 **  (4) load / edit a source file                **
 **  (5) input editor name via popup shell        **
 **  (6) show source file via list widget         **
 **  (7) set / remove breakpoints                 **
 **  (8) step / over modus implemented            **
 **  (9) added command widget "Goto line"         **
 ** (10) added in-/out widgets                    **
 ** (11) added "goto line" modus                  **
 ** (12) "add watch" implemented                  **
 ** (13) "delete watch" completed                 **
 ** (14) added window headers                     **
 ** (15) added horizontal scrollbar in the list   **
 ** (16) found and added a function to adjust     **
 **      the scrollbars in the list widget        **
 ** (17) implemented the help window              **
 ** (18) implemented the PE's structure window    **
 **      (circle)                                 **
 ** (19) implemented the PE's structure window    **
 **      (grid)                                   **
 ** (20) added the PE dimensions in the topology  **
 **      window                                   **
 ** (21) added colours reflecting vector values   **
 **      in the topology window                   **
 ** (22) handling of READ-Instructions improved   **
 **      (program output is now unbuffered)       **
 ** (23) global checkings and corrections         **
 ** (24) added a button for changing the position **
 **      of the first PE in a grid display        **
 ** (25) added Input-Event-Handler                **
 **************************************************/

/* Include standard files */

#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <dirent.h>
#include <ctype.h>
#include <string.h>
#include <signal.h>
#include <math.h>
#include <stdlib.h>

/* defines and includes for the PIPES */

#define READ 0                      /* Reading from a pipe = 0 */
#define WRITE 1                     /* Writing to a pipe   = 1 */
#define BSIZE 10000                 /* String length for one transmission */
#define DIM1 5000                   /* Buffer dimensions for getting a PARZ message */
#define DIM2 80
#define F1 100000                   /* Lines and columns for Parallaxis source files */
#define F2 10000
#define ELENGTH 100                 /* Length of the editor string */
#define FNAMELENGTH 150             /* Length of the fullname string */
#define WAT1 50                     /* Array dimensions for watch expressions */
#define WAT2 30
#define FONT_HT 12.6                /* Font offsets */
#define FONT_OFF (-20)
#define R 200.0                     /* For circle topology */

/* Include own headers : file functions and misc (from Claus Brenner) */

#include "misc.h"
#include "direc.h"

/* Include Xt and Xaw */

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Shell.h>
#include <X11/Xaw/Label.h>
#include <X11/Xaw/Command.h>
#include <X11/Xaw/Box.h>
#include <X11/Xaw/Form.h>
#include <X11/Xaw/AsciiText.h>
#include <X11/Xaw/Dialog.h>
#include <X11/Xaw/Text.h>
#include <X11/Xaw/Viewport.h>
#include <X11/Xaw/List.h>
#include <X11/Xaw/Scrollbar.h>

extern char  puffer[][DIM2];             /* Global buffer */
extern int   p1[2],p2[2],p3[2];          /* Pipes for Read-Write */
extern int   pid,count,rval;             /* Child-ID, bytes read from pipe, return value from system calls */
extern char  inbuf[BSIZE];               /* buffer for PIPE-reading */
extern PopupFisWindow();                 /* Create file select box */
extern clearbuffer();                    /* Clears the puffer[][] */
extern int scanner();                    /* Scans buffer and collect the tokens */
extern char *Fallback_Resources[];       /* Fallback-Ressources if there exists no ressource file */
extern int lastlines;                    /* Lines of the last read Parallaxis file */
extern int re1,re2;                      /* File descriptors of the two pipes */

char pes[100000][10];                    /* buffer for connections */
char *filebuf[F1];                       /* buffer for a source file */
int bpoint[F1];                          /* array for set/remove breakpoints */
char editorstring[ELENGTH];
char fullname[FNAMELENGTH];
char oldfilename[FNAMELENGTH];           /* Editorname and complete filename */
int aktl,atokens;                        /* Actual line in STEP/OVER - modus, Tokens read in the last */
                                         /*                                   pascan1 or pascan2 call */
char *outst;                             /* String for output widget */
BOOLEAN fileok = FALSE;                  /* A Parallaxis source file has been successfully loaded */
char watches[WAT1][WAT2];                /* Array with all watch variables */
int wpoints = 0;                         /* Counter for watch vars */
long lastpe = 0;                         /* Number of tokens of a CONNECTION */
char actproc[100000];                    /* String with active processors */
int dmode = 1;                           /* Drawmode processor topology, Circle = 1, Grid = 2 */
BOOLEAN dimmode = FALSE,                
        colormode = FALSE;               /* PE-labels on / off, Colors on / off */
BOOLEAN startmode = FALSE;               /* Grid topology start lower left or upper left */
int anz_dims, anz_dims2;                 /* Number of entries in the dimension array */
char colorexp[100];                      /* Expression to draw as colours */
char pecolor[100000][100],
     dims[100000][100],
     dims2[100000][100];                 /* Arrays for collecting the tokens */
long left = -1, right = 1;               /* Boundaries for the colormap */
double dd;                               /* Variable distance --> Colormap */
BOOLEAN topon = FALSE;                   /* Topology is on / off */
BOOLEAN WaitingForInput = FALSE;         /* Program is simulated or is waiting for input */
int emode = 0;                           /* Program must be compiled after editing */
XtInputId id1,id2;                       /* Input event handlers when program is simulated */
BOOLEAN id1on=FALSE,id2on=FALSE,
        id1gon=FALSE,id2gon=FALSE,
        id1son=FALSE, id2son=FALSE;      /* Booleans for reflecting that the event handlers are installed */
int lines;                               /* line numbers of a source file */
BOOLEAN debugon = FALSE;                 /* Debug variable. If TRUE there will be various debug outputs to STDERR */

/* Widgets, ApplicationContext, Display, Screen, Graphics Context, Fonts, Colours */

XtAppContext ApplicationContext;
Widget       toplevel,goodbye,load,ctrlmain,ctrlbutt,ctrlfile,ctrlpes,ctrlstruc,fout,fin,message;
Widget       edit,editor,go,step,over,gline,addwatch,delwatch,dialog,diaform,done,pshell;
Widget       listvp,listfile,outtext,intext,petext,mfile,minput,moutput,mwatches,mpes,helpbutton,helppopup,helpdone;
Widget       helpform,helptext,stopex,clearbp,topol,wpopup,wform,wdialog,wdone,glpopup,glform,gldialog,gldone;
Widget       tpopup,tform,td1,td2,td3,tlabel,tstart,tcircle,tgrid,tnothing;
Display      *MyDisplay;
int          MyScreen;
GC           NormalGC;
Font         ft;
int          minc, maxc;

/**************************************************************
 ** FUNCTION: WriteText                                      **
 ** PURPOSE:  Creates text output in various label-widgets   **
 ** INPUTS:   Widget w  - widgets to be changed              **
 **           string st - string to be written               **
 ** RETURNS:  none                                           **
 *************************************************************/

void WriteText(w,st)
     Widget w;
     string st;
{
  Dimension width, height;
  
  XtVaGetValues(w,
		XtNwidth, &width,
		XtNheight, &height,
		NULL );
  XtVaSetValues(w, XtNlabel, (String)st,
		XtNwidth, width,
		XtNheight, height,
		NULL);
  ExposeWidget(w);
}

/***************************************************
 ** FUNCTION: Quit                                **
 ** PURPOSE:  Callback function widget goodbye    **
 **           Exits PGUI                          **
 ** INPUTS:   Xt                                  **
 ** RETURNS:  none                                **
 **************************************************/

void Quit(w, client_data, call_data)
     Widget w;
     XtPointer client_data, call_data;
{
  int i;

  free(outst);
  for (i = 0;i < lastlines; i++)
    free(filebuf[i]);
  paprint("q\n");
  exit(0);
}

/***********************************************
 ** FUNCTION: Editor                          **
 ** PURPOSE:  Callback function widget editor **
 **           Pops up the input shell         **
 ** INPUTS:   Xt                              **
 ** RETURNS:  none                            **
 **********************************************/

void Editor(w, client_data, call_data)
     Widget w;
     XtPointer client_data, call_data;
{
  Widget ctrl = (Widget) client_data;
  Position x,y;
  Dimension width, height;

  XtVaGetValues(ctrl, XtNwidth, &width, XtNheight, &height, NULL);
  XtTranslateCoords (ctrl, (Position) width/2, (Position) height/2, &x, &y);
  XtVaSetValues(pshell, XtNx, x, XtNy, y, NULL);
  XtVaSetValues(dialog, XtNstring, editorstring, NULL); 
  XtSetSensitive (editor, FALSE);
  XtPopup(pshell, XtGrabNonexclusive);
}

/*********************************************
 ** FUNCTION: Edit                          **
 ** PURPOSE:  Callback function widget edit **
 **           Edit the Parallaxis file with **
 **           a system call                 **
 ** INPUTS:   Xt                            **
 ** RETURNS:  none                          **
 ********************************************/

void Edit(w, client_data, call_data)
     Widget w;
     XtPointer client_data, call_data;
{
  char all[200];
  strcpy(all,editorstring);
  strcat(all,fullname);
  system(all);

  emode = 1;
  FisLoad();
  XClearWindow(MyDisplay, XtWindow(ctrlstruc));  
  emode = 0;
}

/**************************************************
 ** FUNCTION: DialogDone                         **
 ** PURPOSE:  Callback function widget dialog    **
 **           Pops down input shell and gets     **
 **           new editor path and name           **
 ** INPUTS:   Xt                                 **
 ** RETURNS:  none                               **
*************************************************/

void DialogDone(w, client_data, call_data)
     Widget w;
     XtPointer client_data, call_data;
{
  char *s,s1[200];
  XtPopdown(pshell);
  XtSetSensitive( editor, TRUE);
  XtVaGetValues(dialog, XtNstring, &s, NULL);
  strcpy(s1,s);
  if (s1[0] != '\0')
    {
      strcpy (editorstring, s1);
      strcpy (s1, "Selected editor: ");
      strcat(s1, editorstring);
      if (editorstring[strlen(editorstring)] != ' ')
	strcat(editorstring," ");
      WriteText(message,s1);
    }
  else
    {
      WriteText(message,"WARNING -- No editor selected !");
    }
}

/*****************************************************
 ** FUNCTION: WriteOut                              **
 ** PURPOSE:  Adds new string to the output widget  **
 ** INPUTS:   char s[] - string to be added         **
 ** RETURNS:  none                                  **
 ****************************************************/

void WriteOut(s)
     char s[];
{
  XtVaSetValues(outtext, XtNstring, s, NULL);
}

/*************************************************************
 ** FUNCTION: CheckFor13                                    **
 ** PURPOSE:  Deletes any ASCII-Char $13 from the buffer    **
 **           and checks for too many blanks.               **
 **           Necessary because an ASCIITextWidget shows    **
 **           a "^M" when a ASCII $13 appears.              **
 ** INPUTS:   char s[] - string to be checked               **
 ** RETURNS:  none                                          **
 ************************************************************/

void CheckFor13(s)
     char s[];
{
  int i,j,k;
  
  k = strlen(s);
  if (k >= 2)
    {
      while ((s[k] == ' ') || (s[k] == '\n') || (s[k] == 13) || (s[k] == '\0'))
	k--;
      s[++k] = '\n';
      s[++k] = '\0';
    }
  
  for (i = 0; i < strlen(s); i++)
    {
      switch (s[i])
	{
	case 13:
	  {
	    s[i] = ' ';
	    break;
	  }
	default:
	  break;
	}
    }
}

/*******************************************************
 ** FUNCTION: WatchOut                                **
 ** PURPOSE:  Produces output of all watch variables  **
 ** INPUTS:   none                                    **
 ** RETURNS:  none                                    **
 ******************************************************/

void WatchOut()
{
  char t[100],*temp;
  int i,j,k,l;
  BOOLEAN fertig;
  
  temp = (char *)calloc(2,sizeof(char));
  strcpy(temp,"\0");
  for (i = 1; i <= wpoints; i++)
    {
      strcpy(t,"EXAMINE ");
      strcat(t,watches[i-1]);
      strcat(t,"\n");
      paprint(t);
      
      strcpy(puffer[0],"\0");
      fertig = FALSE;
      while (!fertig)
	{
	  pascan1(inbuf,TRUE);
	  pascan2(inbuf,FALSE);
	  temp = (char *)realloc(temp,strlen(temp)+strlen(inbuf)+1);
	  strcat(temp,inbuf);
	  if ((strcasecmp(puffer[0],"P*") == 0) || (strcasecmp(puffer[0],"P>") == 0) ||
              (strcasecmp(puffer[0],"[error]") == 0))
	    fertig = TRUE;
	}
      
      CheckFor13(temp);
      temp = (char *)realloc(temp,strlen(temp)+85);
      strcat(temp,"---------------------------------------------------------------------------------\n");
    }
  
  XtVaSetValues(petext, XtNstring, temp, NULL);
  if (!free(temp))
    fprintf(stderr,"WARNING - Couldn't free TEMP !\n");;
}

/********************************************************************
 ** FUNCTION: ScanOut                                              **
 ** PURPOSE:  Scans stdout pipe and adds text to the output widget **
 ** INPUTS:   none                                                 **
 ** RETURNS:  none                                                 **
 *******************************************************************/

int ScanOut(ss,mode)
int ss,mode;
{
  int i,j;
  char t[10];
  
  if (ss)
    if (ss == 12)
      pascan2(inbuf,FALSE);
    else
      if (mode == 2)
	pascan2(inbuf,FALSE);
      else
	pascan1(inbuf,FALSE);
  
  for (i = 0; i < strlen(inbuf); i++)
    {
      switch(inbuf[i])
	{
	case 13: 
	  {
	    for (j = i; j < strlen(inbuf); j++)
	      inbuf[j] = inbuf[j+1];
	    inbuf[j] = '\0';
	    i--;
	    break;
	  }
	case 10: inbuf[i] = '\n'; break;
	default: break;
	}
    }
  
  for (i = 0; i < 5; i++)
    if (inbuf[i] == '[')
      {
	scopy(inbuf,t,i+1,i+7);
	if (strncasecmp(t,"[error]",7) == 0)
	  return(1);
      }

  if (inbuf[0] != '\0')
    {
      outst = (char *)realloc(outst,strlen(outst)+strlen(inbuf)+1);
      strcat(outst,inbuf);
      WriteOut(outst);
    }
  return (0);
}

/*************************************************
 ** FUNCTION: TLabel                            **
 ** PURPOSE:  Callback function widget tlabel   **
 ** INPUTS:   Xt                                **
 ** RETURNS:  none                              **
 ************************************************/

void TLabel(w, client_data, call_data)
     Widget w;
     XtPointer client_data, call_data;
{
  if (!dimmode)
    {
      dimmode = TRUE;
      XtVaSetValues(tlabel, XtNlabel, "with PE-labels", NULL);
    }
  else
    {
      dimmode = FALSE;
      XtVaSetValues(tlabel, XtNlabel, "without PE-labels", NULL);
    }
  ExposeWidget(tlabel);
}

/*************************************************
 ** FUNCTION: TStart                            **
 ** PURPOSE:  Callback function widget tstart   **
 ** INPUTS:   Xt                                **
 ** RETURNS:  none                              **
 ************************************************/

void TStart(w, client_data, call_data)
     Widget w;
     XtPointer client_data, call_data;
{
  if (!startmode)
    {
      startmode = TRUE;
      XtVaSetValues(tstart, XtNlabel, "start upper left", NULL);
    }
  else
    {
      startmode = FALSE;
      XtVaSetValues(tstart, XtNlabel, "start lower left", NULL);
    }
  ExposeWidget(tstart);
}

/********************************************
 ** FUNCTION: CutInput                     **
 ** PURPOSE:  Collects values from inbuf   **
 ** INPUTS:   char[][] - array with values **
 ** OUTPUTS:  none (array is filled in)    **
 *******************************************/

void CutInput(field,allinbuf)
     char field[][100];
     char allinbuf[];
{
  int i=0, k=0, l=0;

  while (allinbuf[i++] != '\n');
  while (allinbuf[i++] != '\n');
  while (i < strlen(allinbuf))
    {
      switch (allinbuf[i])
	{
	case ' ':
	  while (field[k][l] != '\0')
	    field[k][l++] = '\0';
	  k++;
	  l = 0;
	  while (allinbuf[i] == ' ')
	    i++;
	  break;
	case '\n':
	  while (allinbuf[++i] != '\n');
	  i++;
	  break;
	default:
	  field[k][l++] = allinbuf[i];
	  i++;
	  break;
	}
    }
}

/********************************************************************************************
 ** FUNCTION: DrawPes                                                                      **
 ** PURPOSE:  Checks for any existing PE-connection and draws it in the appropriate window **
 ** INPUTS:   long anz_pes   - number of PEs                                               **
 **           char actproc[] - string with length anz_pes, '1' = active, '0' = inactive    **
 ** RETURNS:  none                                                                         **
 *******************************************************************************************/

void DrawPes(anz_pes,actproc)
     long anz_pes;
     char actproc[];
{
  BOOLEAN erfolg, fertig, nocon, neu;
  int i,j,k,x,y,l,m1,m2,seite,ax,ay,ofst,ofst2,abz = 0;
  char t[100],t2[100];
  double alpha;
  int kx[100000], ky[100000];
  char tes[10],pesend[100];
  int cl;
  BOOLEAN colerror = TRUE, ready = FALSE;
  char dimarr[100000];
  int sl;
  char hl;
  double hf;
  int z1;
  BOOLEAN bl = FALSE;
  int offs = 0;

  if (topon)
    if (anz_pes > 10000)
      {
	XDrawString( MyDisplay, XtWindow(ctrlstruc), NormalGC,
		    140,200,"Too many Processor elements !",29);
	return;
      }
  
  erfolg = FALSE;
  fertig = FALSE;
  nocon = FALSE;
  neu = FALSE;

  if (!topon)
    return;

  paprint("CONNECTIONS\n");                /* send CONNECTIONS to PARZ */
  count = -1;
  
  while (!fertig)
    {
      while ((!erfolg) || (count != -1))
	{
	  pascan2(inbuf,TRUE);
	  
	  if (count != -1)
	    {
	      for (i = 0; i <= atokens; i++)            /* fill in the tokens */
		{
		  strcpy(pes[i+lastpe],puffer[i]);
		}
	      lastpe += atokens;
	      erfolg = TRUE;
	    }
	}
      
      if (strcasecmp(puffer[0],"no") == 0)              /* no connections exist */
	nocon = TRUE;
      
      pascan1(inbuf,TRUE);
      if ((strcasecmp(puffer[0],"P*") == 0) || (strcasecmp(puffer[0],"P>") == 0))
	fertig = TRUE;
    }
  
  XClearWindow(MyDisplay, XtWindow(ctrlstruc));
  strcpy(puffer[0],"\0");

  if (dimmode)                            /* draw the PE dimensions if possible */
    {
      paprint("EXAMINE DIM1\n");          /* DIM1 */
      strcpy(dimarr,"\0");
      
      ready = FALSE;
      while (!ready)
	{
	  pascan1(inbuf, TRUE);
	  pascan2(inbuf, FALSE);
	  strcat(dimarr,inbuf);
          if ((strcasecmp(puffer[0],"P*") == 0) || (strcasecmp(puffer[0],"P>") == 0) ||
              (strcasecmp(puffer[0],"[error]") == 0))
            ready = TRUE;
	}

      scopy(dimarr,tes,2,8);

      if (strcasecmp(tes,"[error]") != 0) /* no error */
	{
	  anz_dims = 1;                   /* o.k. */
	  CutInput(dims,dimarr);          /* collect the required values from the inbuf string */
	}
      else
	anz_dims = 0;                     /* error : not in a parallel block */
      
      paprint("EXAMINE DIM2\n");
      strcpy(dimarr,"\0");
      strcpy(puffer[0],"\0");
      
      ready = FALSE;
      while (!ready)
	{
	  pascan1(inbuf, TRUE);
	  pascan2(inbuf, FALSE);
	  strcat(dimarr,inbuf);
          if ((strcasecmp(puffer[0],"P*") == 0) || (strcasecmp(puffer[0],"P>") == 0) ||
              (strcasecmp(puffer[0],"[error]") == 0))
            ready = TRUE;
	}
      
      scopy(dimarr,tes,2,8);
      
      if (strcasecmp(tes,"[error]") != 0) /* no error */
	{
	  anz_dims2 = 1;                  /* o.k. */
	  CutInput(dims2,dimarr);         /* collect the required values from the inbuf string */
	}
      else
	anz_dims2 = 0;                    /* error : not in a parallel block */
    }
  
  if (colormode)                          /* colormode = TRUE, PE's had to be drawn in the color according to */
    {                                     /*                   their vector value from "colorexp"             */
      strcpy(pesend,"EXAMINE ");
      strcat(pesend,colorexp);
      strcat(pesend,"\n");

      paprint(pesend);

      strcpy(dimarr,"\0");
      
      strcpy(puffer[0],"\0");
      
      ready = FALSE;
      while (!ready)
	{
	  pascan1(inbuf, TRUE);
	  pascan2(inbuf, FALSE);
	  strcat(dimarr,inbuf);
          if ((strcasecmp(puffer[0],"P*") == 0) || (strcasecmp(puffer[0],"P>") == 0) ||
              (strcasecmp(puffer[0],"[error]") == 0))
            ready = TRUE;
	}
      
      scopy(dimarr,tes,2,8);
      
      if (strcasecmp(tes,"[error]") != 0)
	{
	  CutInput(pecolor,dimarr);
	  colerror = FALSE;
	}
      else
	colerror = TRUE;

      if ((right - left) != 0)
	dd = (double) ((double)(right - left) / (double)(maxc - minc));
      else
	dd += 0.01;

      for (z1 = minc; z1 <= maxc; z1++)
	{
	  XSetForeground(MyDisplay, NormalGC, z1);
	  XFillRectangle(MyDisplay, XtWindow(ctrlstruc), NormalGC, (z1-minc)*2+13, 0, 2, 5);
	}
    }
  
  if (dmode == 1)                         /* draw mode = circle */
    {
      /* Circle topology */
      
      for (i = 0; i <= (anz_pes-1); i++)
	{
	  alpha = i * 2 * PI / anz_pes;            
	  x = 242 + (int)(R * sin(alpha));
	  y = 212 - (int)(R * cos(alpha));
	  kx[i] = x+5;
	  ky[i] = y+5;
	  if (actproc[i] == '0')
	    {
	      XSetForeground(MyDisplay, NormalGC, 1);
	      XDrawRectangle(MyDisplay, XtWindow(ctrlstruc), NormalGC, x, y, 10, 10);
	      abz++;
	    }
	  else
	    {
	      XSetForeground(MyDisplay, NormalGC, 1);
	      XDrawRectangle(MyDisplay, XtWindow(ctrlstruc), NormalGC, x-1, y-1, 11, 11);
	      if (!colerror)
		{
		  if (strncasecmp(pecolor[i-abz],"TRUE",4) == 0)
		    {
		      cl = maxc;
		      bl = TRUE;
		    }
		  else
		    if (strncasecmp(pecolor[i-abz],"FALSE",5) == 0)
		      {
			cl = minc;
			bl = TRUE;
		      }
		    else
		      {
			hl = pecolor[i-abz+offs][0];
			hf = atof(pecolor[i-abz+offs]);
			if (hf != 0)
			  {
			    cl = minc + (int) ((hf - left) / dd);
			    if (hf > right)
			      cl = maxc;
			    else
			      if (hf < left)
				cl = minc;
			  }
	                else
			  if ((isdigit(hl)) || (hl == '.') || (hl == '-'))
			    {
			      cl = minc + (int) ((hf - left) / dd);
			      if (hf > right)
				cl = maxc;
			      else
				if (hf < left)
				  cl = minc;
			    }
			  else
			    {
			      cl = 1;
			      offs += 2;
			    }
		      }
	        }
	      else
		cl = 0;
	      
              XSetForeground(MyDisplay, NormalGC, cl);
	      XFillRectangle(MyDisplay, XtWindow(ctrlstruc), NormalGC, x, y, 10, 10);
	      if ((dimmode) && (anz_dims == 1))
		{
		  XSetForeground(MyDisplay, NormalGC, 1);
		  XDrawString(MyDisplay, XtWindow(ctrlstruc), NormalGC, x+10, y+20, dims[i-abz], strlen(dims[i-abz]));
		  if (anz_dims2 == 1)
		    XDrawString(MyDisplay, XtWindow(ctrlstruc), NormalGC, x+10, y+30, dims2[i-abz],
				strlen(dims2[i-abz]));	      
		}
	    }
	}
    }
  else
    {
      /* Grid topology */
      
      seite = (int) (sqrt((double) anz_pes) + 0.99);
      ax = (int) 440/(seite-1);
      ay = (int) 350/(seite-1);
      ofst = (int) (ay / 5);
      ofst2 = (int) (ofst / 2);
      for (i = 0; i < seite; i++)
	{
	  for (j = 0; j < seite; j++)
	    {
	      if ((j + (i * seite)) + 1 <= anz_pes)
		{
		  y = 390 - (i * ay);
		  if (startmode)                     /* Start upper left */
		    y = -y+440;
		  x = 15 + (j * ax);
		  kx[j + (i*seite)] = x+5;
		  ky[j + (i*seite)] = y+5;
		  if (actproc[j + (i*seite)] == '0')
		    {
		      XSetForeground(MyDisplay, NormalGC, 1);
		      XDrawRectangle(MyDisplay, XtWindow(ctrlstruc), NormalGC, x, y, 10, 10);
		      abz++;
		    }
		  else
		    {
		      XSetForeground(MyDisplay, NormalGC, 1);
		      XDrawRectangle(MyDisplay, XtWindow(ctrlstruc), NormalGC, x-1, y-1, 11, 11);
		      if (!colerror)
			{
			  if (strncasecmp(pecolor[j + (i * seite) - abz],"TRUE",4) == 0)
			    {
			      cl = maxc;
			      bl = TRUE;
			    }
			  else
			    if (strncasecmp(pecolor[j + (i * seite) - abz],"FALSE",5) == 0)
			      {
				cl = minc;
				bl = TRUE;
			      }
			    else
			      {
				hl = pecolor[j + (i * seite) - abz + offs][0];
				hf = atof(pecolor[j + (i * seite) - abz + offs]);
				if (hf != 0)
				  {
				    cl = minc + (int) ((hf - left) / dd);
				    if (hf > right)
				      cl = maxc;
				    else
				      if (hf < left)
					cl = minc;
				  }
				else
				  if ((isdigit(hl)) || (hl == '.') || (hl == '-')) 
				    {
				      cl = minc + (int) ((hf - left) / dd);
				      if (hf > right)
					cl = maxc;
				      else
					if (hf < left)
					  cl = minc;
				    }
				  else
				    {
				      cl = 1;
				      offs += 2;
				    }
			      }
			}
		      else
			cl = 0;

		      XSetForeground(MyDisplay, NormalGC, cl);
		      XFillRectangle(MyDisplay, XtWindow(ctrlstruc), NormalGC, x, y, 10, 10);
		      if ((dimmode) && (anz_dims == 1))
			{
			  XSetForeground(MyDisplay, NormalGC, 1);
			  XDrawString(MyDisplay, XtWindow(ctrlstruc), NormalGC, x+10, y+20,
				      dims[j + (i * seite) - abz], strlen(dims[j + (i * seite) - abz]));
			  if (anz_dims2 == 1)
			    XDrawString(MyDisplay, XtWindow(ctrlstruc), NormalGC, x+10, y+30,
					dims2[j + (i * seite) - abz], strlen(dims2[j + (i * seite) - abz]));
			}
		    }
		}
	    }
	}
    }

  /* Draw connections between PE's */

  XSetForeground(MyDisplay, NormalGC, 1);
  i = 0;
  if (!nocon)
    {
      while (i < lastpe)
	{
	  while ((strcasecmp(pes[i],"TO") != 0) && (i < lastpe))
	    i++;
	  
	  if (i < lastpe)
	    {
	      m1 = atoi(pes[i-2]) - 1;
	      m2 = atoi(pes[i+1]) - 1;
	      if (m1 > m2)
		OrderInt(&m1,&m2);
	      if (m1 != m2)
		{
		  XDrawLine(MyDisplay, XtWindow(ctrlstruc), NormalGC,
			    kx[m1], ky[m1],
			    kx[m2], ky[m2]);
		  if (dmode == 2)                /* <-- only Grid mode */
		    {
		      if (((ABS(m2 - m1)) > 1) && (ky[m1] == ky[m2]))      /* Torus */
			{
			  ofst += (int)(((m2 % seite)*2) % (ay-13)); 
			  XDrawLine(MyDisplay, XtWindow(ctrlstruc), NormalGC,
				    kx[m1],ky[m1],
				    kx[m1] + ofst2, ky[m1] - ofst);
			  XDrawLine(MyDisplay, XtWindow(ctrlstruc), NormalGC,
				    kx[m1] + ofst2, ky[m1] - ofst,
				    kx[m2] - ofst2, ky[m2] - ofst);
			  XDrawLine(MyDisplay, XtWindow(ctrlstruc), NormalGC,
				    kx[m2] - ofst2, ky[m2] - ofst,
				    kx[m2], ky[m2]);
			  ofst -= (int)(((m2 % seite)*2) % (ay-13));
			}
		      if (((ABS(m2 - m1)) > seite) && (kx[m1] == kx[m2]))   /* Torus */
			{
			  ofst += (int)(((m2 / seite)*2) % (ax-13));
			  XDrawLine(MyDisplay, XtWindow(ctrlstruc), NormalGC,
				    kx[m1],ky[m1],
				    kx[m1] + ofst, ky[m1] - ofst2);
			  XDrawLine(MyDisplay, XtWindow(ctrlstruc), NormalGC,
				    kx[m1] + ofst, ky[m1] - ofst2,
				    kx[m2] + ofst, ky[m2] + ofst2);
			  XDrawLine(MyDisplay, XtWindow(ctrlstruc), NormalGC,
				    kx[m2] + ofst, ky[m2] + ofst2,
				    kx[m2], ky[m2]);
			  ofst -= (int)(((m2 / seite)*2) % (ax-13));
			}
		    }
		}
	      else
		XDrawArc(MyDisplay, XtWindow(ctrlstruc), NormalGC,
			 kx[m1]-8, ky[m1],
			 16,16,
			 0,360*64);
	      
	      l = i-2;
	      sl = strlen(pes[i+2]) - 1;
	      while (pes[i+2][sl] == ',')
		{
		  i += 2;
		  m1 = atoi(pes[l]) - 1;
		  m2 = atoi(pes[i+1]) - 1;
		  if (m1 > m2)
		    OrderInt(&m1,&m2);
		  if (m1 != m2)
		    {
		      XDrawLine(MyDisplay, XtWindow(ctrlstruc), NormalGC,
				kx[m1], ky[m1],
				kx[m2], ky[m2]);
		      if (dmode == 2)
			{
			  if (((ABS(m2 - m1)) > 1) && (ky[m1] == ky[m2]))                 /* Torus */
			    {
			      ofst += (int)(((m2 % seite)*2) % (ay-13)); 
			      XDrawLine(MyDisplay, XtWindow(ctrlstruc), NormalGC,
					kx[m1],ky[m1],
					kx[m1] + ofst2, ky[m1] - ofst);
			      XDrawLine(MyDisplay, XtWindow(ctrlstruc), NormalGC,
					kx[m1] + ofst2, ky[m1] - ofst,
					kx[m2] - ofst2, ky[m2] - ofst);
			      XDrawLine(MyDisplay, XtWindow(ctrlstruc), NormalGC,
					kx[m2] - ofst2, ky[m2] - ofst,
					kx[m2], ky[m2]);
			      ofst -= (int)(((m2 % seite)*2) % (ay-13)); 
			    }
			  if (((ABS(m2 - m1)) > seite) && (kx[m1] == kx[m2]))             /* Torus */
			    {
			      ofst += (int)(((m2 / seite)*2) % (ax-13)); 
			      XDrawLine(MyDisplay, XtWindow(ctrlstruc), NormalGC,
					kx[m1],ky[m1],
					kx[m1] + ofst, ky[m1] - ofst2);
			      XDrawLine(MyDisplay, XtWindow(ctrlstruc), NormalGC,
					kx[m1] + ofst, ky[m1] - ofst2,
					kx[m2] + ofst, ky[m2] + ofst2);
			      XDrawLine(MyDisplay, XtWindow(ctrlstruc), NormalGC,
					kx[m2] + ofst, ky[m2] + ofst2,
					kx[m2], ky[m2]);
			      ofst -= (int)(((m2 / seite)*2) % (ax-13)); 
			    }
			}
		    }
		  else
		    XDrawArc(MyDisplay, XtWindow(ctrlstruc), NormalGC,
			     kx[m1]-8, ky[m1],
			     16,16,
			     0,360*64);
		  
		  sl = strlen(pes[i+2]) - 1;
		}
	      
	      sl = strlen(pes[i+2]) - 1;
	      i++;
	    }
	}	
    }
  else
    if (dmode == 1)
      XDrawString( MyDisplay, XtWindow(ctrlstruc), NormalGC,
		  180,200,"no connections",14);
    else
      XDrawString( MyDisplay, XtWindow(ctrlstruc), NormalGC,
		  180, 30,"no connections", 14);
  
  strcpy(t,"Processor topology (");
  if (dmode == 1)
    strcat(t,"Circle; ");
  else
    strcat(t,"Grid; ");

  itoa(anz_pes,t2);
  strcat(t,t2);
  strcat(t," PEs)"); 
  if (colormode)
    {
      strcat(t,",(");
      strcat(t,colorexp);
      strcat(t,";");
      if (!bl)
	itoa((int) left,t2);
      else
	strcpy(t2,"FALSE");
      strcat(t,t2);
      strcat(t,";");
      if (!bl)
	itoa((int) right, t2);
      else
	strcpy(t2,"TRUE");
      strcat(t,t2);
      strcat(t,")");
    }
  
  WriteText(mpes, t);

  strcpy(inbuf,"\0");
  lastpe = 0;
}

/*************************************************
 ** FUNCTION: TCircle                           **
 ** PURPOSE:  Callback function widget tcircle  **
 ** INPUTS:   Xt                                **
 ** RETURNS:  none                              **
 ************************************************/

void TCircle(w, client_data, call_data)
     Widget w;
     XtPointer client_data, call_data;
{
  char *s1,*s2,*s3;

  XtPopdown(tpopup);

  XtVaGetValues(td1, XtNstring, &s1, NULL);
  XtVaGetValues(td2, XtNstring, &s2, NULL);
  XtVaGetValues(td3, XtNstring, &s3, NULL);

  dmode = 1;
  topon = TRUE;
  if (s1[0] == '\0')
    {
      colormode = FALSE;
      strcpy(colorexp,"\0");
    }
  else
    {
      colormode = TRUE;
      strcpy(colorexp, s1);
      if (s2[0] != '\0')
	left = atol(s2);
      if (s3[0] != '\0')
	right = atol(s3);
    }
  DrawPes(strlen(actproc), actproc);
}

/***********************************************
 ** FUNCTION: TGrid                           **
 ** PURPOSE:  Callback function widget tgrid  **
 ** INPUTS:   Xt                              **
 ** RETURNS:  none                            **
 **********************************************/

void TGrid(w, client_data, call_data)
     Widget w;
     XtPointer client_data, call_data;
{
  char *s1,*s2,*s3;

  XtPopdown(tpopup);

  XtVaGetValues(td1, XtNstring, &s1, NULL);
  XtVaGetValues(td2, XtNstring, &s2, NULL);
  XtVaGetValues(td3, XtNstring, &s3, NULL);

  dmode = 2;
  topon = TRUE;
  if (s1[0] == '\0')
    {
      colormode = FALSE;
      strcpy(colorexp,"\0");
    }
  else
    {
      colormode = TRUE;
      strcpy(colorexp, s1);
      if (s2[0] != '\0')
	left = atol(s2);
      if (s3[0] != '\0')
	right = atol(s3);
    }
  DrawPes(strlen(actproc),actproc);
}

/*************************************************
 ** FUNCTION: TNothing                          **
 ** PURPOSE:  Callback function widget tnothing **
 ** INPUTS:   Xt                                **
 ** RETURNS:  none                              **
 ************************************************/

void TNothing(w, client_data, call_data)
     Widget w;
     XtPointer client_data, call_data;
{
  XtPopdown(tpopup);
  XClearWindow(MyDisplay, XtWindow(ctrlstruc));

  strcpy(colorexp,"\0");
  colormode = FALSE;
  topon = FALSE;
  dimmode = FALSE;
  
  XtVaSetValues(tlabel, XtNlabel, "without PE-labels",NULL);
  WriteText(mpes,"Processor topology");
}

/**********************************************
 ** FUNCTION: Topology                       **
 ** PURPOSE:  Callback function widget topol **
 ** INPUTS:   Xt                             **
 ** RETURNS:  none                           **
 *********************************************/

void Topology(w, client_data, call_data)
     Widget w;
     XtPointer client_data, call_data;
{
  char *s = "\0";
  char t1[100],t2[100];
  Position x,y;
  Dimension width, height;
  Dimension w1,h1;

  if (!fileok)
    {
      WriteText(message,"ERROR - No correct source file loaded !!!");
      return;
    }
  
  XtVaGetValues( ctrlbutt, XtNwidth, &width, XtNheight, &height, NULL);
  XtTranslateCoords (ctrlbutt, (Position) width/4, (Position) height/2, &x, &y);
  XtVaSetValues( tpopup, XtNx, x, XtNy, y, NULL);
  XtVaSetValues( td1, XtNstring, colorexp, NULL);
  itoa((int) left, t1);
  XtVaSetValues( td2, XtNstring, t1, NULL);
  itoa((int) right, t2);
  XtVaSetValues( td3, XtNstring, t2, NULL);
  XtVaGetValues( tlabel, XtNwidth, &w1, XtNheight, &h1, NULL);
  if (dimmode)
    XtVaSetValues( tlabel, XtNlabel, "with PE-labels", XtNwidth, w1, XtNheight, h1, NULL);
  else
    XtVaSetValues( tlabel, XtNlabel, "without PE-labels", XtNwidth, w1, XtNheight, h1, NULL);
  XtPopup(tpopup, XtGrabNonexclusive);
}

/****************************************************************
 ** FUNCTION: Search                                           **
 ** PURPOSE:  Checks if the current line is a READ-Instruction **
 ** INPUTS:   none                                             **
 ** RETURNS:  none                                             **
 ***************************************************************/

BOOLEAN Search()
{
  int i;
  char test[200];
  
  if (aktl == 0)
    return(FALSE);

  for (i = 0; i < (strlen(filebuf[aktl-1]) - 3); i++)
    {
      scopy(filebuf[aktl-1],test,i,i+3);
      if (strcasecmp(test,"read") == 0)
	return (TRUE);
    }
  return (FALSE);
}

/*******************************************
 ** FUNCTION: Go                          **
 ** PURPOSE:  Callback function widget go **
 ** INPUTS:   Xt                          **
 ** RETURNS:  none                        **
 ******************************************/

void Go(w, client_data, call_data)
     Widget w;
     XtPointer client_data, call_data;
{
  Widget lf = (Widget) client_data;
  int i,j,k  ,l2;
  BOOLEAN erfolg, fertig, endof;
  char t[50],t1[5];
  int anz_read = 0,tok,warn = 0;
  Position shown;
  char dummy[BSIZE];

  erfolg = FALSE;
  fertig = FALSE;
  endof  = FALSE;

  if (!WaitingForInput)
    paprint("GO\n");

  WriteText(message,"RUNNING");

  XtSetSensitive(step, FALSE);
  XtSetSensitive(over, FALSE);
  XtSetSensitive(gline,FALSE);
  XtSetSensitive(listfile,FALSE);
  XtSetSensitive(load,FALSE);
  XtSetSensitive(topol,FALSE);
  XtSetSensitive(addwatch, FALSE);
  XtSetSensitive(delwatch, FALSE);
  XtSetSensitive(go,FALSE);
  XtSetSensitive(edit,FALSE);
  XtSetSensitive(clearbp,FALSE);

  ScanOut(1,2);

  while (!fertig)
    {
      while (!erfolg)
	{
	  count = -1;
	  k = 0;
	  while ((count == -1) && (k < 5000))
	    {
	      pascan1(inbuf,TRUE);
	      k++;
	    }
	  
	  if (k < 5000)
	    {
	      while ((strcasecmp(puffer[atokens],"P*") != 0) && (strcasecmp(puffer[atokens],"P>") != 0))
		{
		  pascan1(dummy,TRUE);
		  if (count != -1)
		    strcat(inbuf,dummy);
		}
	    }
	  
	  clearbuffer();
	  tok = scanner(inbuf);

	  if (debugon)
	    fprintf(stderr,"INBUF(GO) = %s\n",inbuf);

	  if (k >= 5000)
	    {
	      count = -1;
	      ScanOut(1,2);
	      WaitingForInput = TRUE;
	      
	      id2 = XtAppAddInput (ApplicationContext,re2,XtInputReadMask,ScanOut,12);
	      id1 = XtAppAddInput (ApplicationContext,re1,XtInputReadMask,Go,listfile);
	      id1on = TRUE;
	      id2on = TRUE;
	      return;
	    }
	  else
	    WaitingForInput = FALSE;
	  
	  if ((strncasecmp(inbuf,"program is not executable",25) == 0) || (!fileok))
	    {
	      WriteText(message,"ERROR - No correct source file loaded !!!");
	      XtSetSensitive(step, TRUE);
	      XtSetSensitive(over, TRUE);
	      XtSetSensitive(gline,TRUE);
	      XtSetSensitive(listfile, TRUE);
	      XtSetSensitive(load, TRUE);
	      XtSetSensitive(topol, TRUE);	      
	      XtSetSensitive(addwatch, TRUE);
	      XtSetSensitive(delwatch, TRUE);
	      XtSetSensitive(go,TRUE);
	      XtSetSensitive(edit,TRUE);
	      XtSetSensitive(clearbp,TRUE);
	      erfolg = TRUE;
	      fertig = TRUE;
	      endof = FALSE;
	      return;
	    }

	  for (i = 0; i <= tok; i++)
	    {
	      if (strcasecmp(puffer[i],"[warning]") == 0)
		warn = 1;
	      if (strcasecmp(puffer[i],"P>") == 0)
		{
		  if (id1on)
		    {
		      XtRemoveInput(id1);
		      id1on = FALSE;
		    }
		  if (id2on)
		    {
		      XtRemoveInput(id2);
		      id2on = FALSE;
		    }
		  if (warn)
		    ScanOut(0,1);
		  erfolg = TRUE;
		  fertig = TRUE;
		  endof  = TRUE;
		  i = tok+1;
		}
	      if (strcasecmp(puffer[i],"source-line") == 0)
		{
		  XtSetSensitive(step, TRUE);
		  XtSetSensitive(over, TRUE);
		  XtSetSensitive(gline,TRUE);
		  XtSetSensitive(listfile, TRUE);
		  XtSetSensitive(load, TRUE);
		  XtSetSensitive(topol, TRUE);	      
		  XtSetSensitive(addwatch, TRUE);
		  XtSetSensitive(delwatch, TRUE);
		  XtSetSensitive(go,TRUE);
		  XtSetSensitive(edit,TRUE);
		  XtSetSensitive(clearbp,TRUE);
		  anz_read++;
		  aktl = atoi(puffer[i+1]);
		  XawListHighlight(listfile, aktl-1);
		  shown = (Position)(aktl * FONT_HT + FONT_OFF);
		  XawViewportSetCoordinates(listvp, (Position)0, shown);
		  erfolg = TRUE;
		  fertig = TRUE;
		  endof = FALSE;
		  if (strcasecmp(puffer[0],"forced") != 0)
		    strcpy(t,"Arrived at a breakpoint at line ");
		  else
		    strcpy(t,"Interrupted at line ");

		  itoa(aktl,t1);
		  strcat(t,t1);
		  WriteText(message,t);
		  if (id1on)
		    {
		      XtRemoveInput(id1);
		      id1on = FALSE;
		    }
		  if (id2on)
		    {
		      XtRemoveInput(id2);
		      id2on = FALSE;
		    }
		}
	      if (strcasecmp(puffer[i],"processors") == 0)
		{
		  if (warn)
		    ScanOut(0,1);
		  l2 = i + 2;
		  strcpy(actproc,puffer[l2]);
		  l2++;
		  while (puffer[l2][0] != 'P')
		    {
		      if (puffer[l2][0] == '\0')
			{
			  pascan1(inbuf,TRUE);
			  l2 = 0;
			}
		      strcat(actproc,puffer[l2]);
		      l2++;
		    }
		}
	    }

	  if ((!erfolg) && (!fertig) && (strcasecmp(puffer[0],"forced") == 0))
	    {
	      erfolg = TRUE;
	      fertig = TRUE;
	      endof  = FALSE;
	      XtSetSensitive(step, TRUE);
	      XtSetSensitive(over, TRUE);
	      XtSetSensitive(gline,TRUE);
	      XtSetSensitive(listfile, TRUE);
	      XtSetSensitive(load, TRUE);
	      XtSetSensitive(topol, TRUE);	      
	      XtSetSensitive(addwatch, TRUE);
	      XtSetSensitive(delwatch, TRUE);
	      XtSetSensitive(go,TRUE);
	      XtSetSensitive(edit,TRUE);
	      XtSetSensitive(clearbp,TRUE);
	      anz_read++;
	      strcpy(t,"Program interrupted but source-line is not available !");
	      WriteText(message,t);
	    }

	  if ((strcasecmp(puffer[0],"P>") == 0) && (anz_read == 0))
	    {
	      erfolg = TRUE;
	      fertig = TRUE;
	      endof = TRUE;
	    }
	  else
	    if ((strcasecmp(puffer[tok],"P*") == 0) && (anz_read != 0))
	      {
		erfolg = TRUE;
		fertig = TRUE;
		endof = FALSE;
	      }
	}
    }

  ScanOut(1,2);
  ScanOut(1,1);
  WatchOut();

  if (!endof)
    DrawPes(strlen(actproc),actproc);
  else
    {
      XtSetSensitive(step, TRUE);
      XtSetSensitive(over, TRUE);
      XtSetSensitive(gline,TRUE);
      XtSetSensitive(listfile, TRUE);
      XtSetSensitive(load, TRUE);
      XtSetSensitive(topol, TRUE);	      
      XtSetSensitive(addwatch, TRUE);
      XtSetSensitive(delwatch, TRUE);
      XtSetSensitive(go,TRUE);
      XtSetSensitive(edit,TRUE);
      XtSetSensitive(clearbp,TRUE);
      if (warn)
	ScanOut(0,1);
      WriteText(message,"Arrived at the end of the source code !");
      XawViewportSetCoordinates(listvp, (Position)0, (Position)(lines * FONT_HT + FONT_OFF));
      XawListHighlight(listfile, lines-1);
      aktl = 0;
      outst = (char *)realloc(outst,2);
      strcpy(outst,"\0");
      XClearWindow(MyDisplay, XtWindow(ctrlstruc));
    }
}

/********************************************************
 ** FUNCTION: CheckForLine                             **
 ** PURPOSE:  Checks the line where the breakpoint     **
 **           will be actually set or removed          **
 ** INPUTS:   none                                     **
 ** RETURNS:  int - Current line number of breakpoint  **
 *******************************************************/

int CheckForLine()
{
  BOOLEAN erfolg, fertig;
  int i,j,k,m,za;
  char t[50],t1[50];
  int tok=0,tok1,tok2;
  char dummy[BSIZE];
  char d2[10000];

  erfolg = FALSE;
  fertig = FALSE;

  count = -1;
  za = 0;
  strcpy(d2,"\0");
  while (!erfolg) 
    {
      while ((!fertig) && (za < 500))
	{
	  pascan2(inbuf,FALSE);
	  za++;
	  strcat(d2,inbuf);
	  pascan1(dummy,TRUE);
	  if ((strcasecmp(puffer[atokens],"P>") == 0) || (strcasecmp(puffer[atokens],"P*") == 0))
	    fertig = TRUE;
	}

      while ((tok < 8) && (strcasecmp(puffer[0],"[error]")))
	{
	  pascan2(inbuf,FALSE);
	  strcat(d2,inbuf);
	  clearbuffer();
	  tok = scanner(d2);
	}

/*      if (za >= 500)
	{
	  WriteText(message,"WARNING - PARZ couldn't set/remove the breakpoint !");
	  return(0);
	}

      clearbuffer();
      tok = scanner(d2);
*/
      if (debugon)
	fprintf(stderr,"INBUF(BREAKPOINT) = %s\n",d2);

      for (i = 0; i <= tok; i++)
	{
	  if (strcasecmp(puffer[i],"statement") == 0)
	    {
	      erfolg = TRUE;
	      k = i+2;
	    }
	  if (strcasecmp(puffer[i],"[error]") == 0)
	    {
	      WriteText(message,"No breakpoint possible in this line !!!");
	      return (0);
	    }
	  if ((i >= tok) && (!erfolg))
	    {
	      WriteText(message,"WARNING - PARZ couldn't set/remove the breakpoint !");
	      return(0);
	    }
	}
    }
  
  if (strcasecmp(puffer[k+1],":") != 0)
    strcat(puffer[k],puffer[k+1]);
  
  reverse(puffer[k]);
  
  i = 0;
  while(puffer[k][i++] != '!');
  i--;
  
  if (puffer[k][0] == 'B')
    scopy(puffer[k],t,2,i);
  else
    scopy(puffer[k],t,1,i);
  
  reverse(t);
  j = atoi(t);
  
  return(j);
}
 
/*************************************************
 ** FUNCTION: Listfile                          **
 ** PURPOSE:  Callback function widget listfile **
 **           Setting or removing a breakpoint  **
 ** INPUTS:   Xt                                **
 ** RETURNS:  none                              **
 ************************************************/

void Listfile(w, client_data, call_data)
     Widget w;
     XtPointer client_data, call_data;
{
  XawListReturnStruct *listret = (XawListReturnStruct *)call_data; 
  char temp[50],t1[50],t2[50],t3[F2],t4[F2];
  int po,aline,i;

  po = (listret->list_index) + 1;

  strcpy(temp, "BREAKPOINT ");
  itoa(po,t1);

  if (bpoint[po] == 1)
    {
      strcat(temp," -");                         /* Remove breakpoint */
      strcat(temp, t1);
      strcat(temp, "\n");
      paprint(temp);

      aline = CheckForLine();

      if (aline > 0)
	{
	  for (i = po; i <= aline; i++)          /* Mark breakpoint: removed */
	    bpoint[i] = 0;

	  if (po > aline)
	    {
	      bpoint[aline] = 0;
	      bpoint[po] = 0;
	    }

	  strcpy(t2,"Breakpoint removed from line ");
	  itoa(aline,t1);
	  strcat(t2, t1);
	  WriteText(message,t2);

	  strcpy(t4,filebuf[aline-1]);
	  scopy(t4,t3,2,strlen(t4));
	  leftcat(" ",t3);
	  strcpy(t4,t3);
	  strcpy(filebuf[aline-1],t4);
	  DrawListFile(lines);
	}
    }
  else
    {
      strcat(temp, t1);
      strcat(temp, "\n");
      paprint(temp);
      
      aline = CheckForLine();

      if (aline > 0)
	{
	  for (i = po; i <= aline; i++)          /* Mark breakpoint: set */
	    bpoint[i] = 1;

	  if (po > aline)
	    {
	      bpoint[aline] = 1;
	      bpoint[po] = 1;
	    }
	  
	  strcpy(t2,"Breakpoint set at line ");
	  itoa(aline,t1);
	  strcat(t2, t1);
	  WriteText(message,t2);
	  
 	  strcpy(t4,filebuf[aline-1]);
	  scopy(t4,t3,2,strlen(t4));
	  leftcat("B",t3);
	  strcpy(t4,t3);
	  strcpy(filebuf[aline-1],t4);
	  DrawListFile(lines);
	}
    }

  XawListUnhighlight(w);
  if ((aktl > 0) && (aktl <= lines))
    XawListHighlight(w,aktl-1);
}

/*********************************************
 ** FUNCTION: GoStep                        **
 ** PURPOSE:  Callback function widget step **
 ** INPUTS:   Xt                            **
 ** RETURNS:  none                          **
 ********************************************/

void GoStep(w, client_data, call_data)
     Widget w;
     XtPointer client_data, call_data;
{
  Widget lf = (Widget) client_data;
  int i,j,k=0,l,lauf,l2,m;
  BOOLEAN erfolg, fertig, fileend;
  char t[50],t1[5];
  int anz_read = 0;
  Position shown;
  char dummy[BSIZE];
  int tok,warn=0;
  
  erfolg = FALSE;
  fertig = FALSE;
  fileend = FALSE;

  paprint("STEP\n");

  WriteOut(outst);

  while (!erfolg)
    {
    hoch:
      count = -1;
      while ((count == -1) && (k < 5000))
	{
	  pascan1(inbuf,TRUE);
	  k++;
	}

      if (k < 5000)
	{
	  while ((strcasecmp(puffer[atokens],"P*") != 0) && (strcasecmp(puffer[atokens],"P>") != 0))
	    {
	      pascan1(dummy,TRUE);	 
	      if (count != -1)
		strcat(inbuf,dummy);
	    }
	}
      
      clearbuffer();
      tok = scanner(inbuf);

      if (debugon)
	fprintf(stderr,"INBUF(STEP) = %s\n",inbuf);

      if (k >= 5000)
	{
	  if (Search())
	    {
	      WriteText(message,"Execution has reached a READ-statement !");
	      XBell(MyDisplay, 100);
	      XtSetSensitive( ctrlbutt, FALSE);
	      pascan1(inbuf,FALSE);
	      return;
	    }
	  k = 0;
	  goto hoch;
	}

      if ((strncasecmp(inbuf,"program is not executable",25) == 0) || (!fileok))
	{
	  WriteText(message,"ERROR - No correct source file loaded !!!");
	  return;
	}
      
      for (lauf = 0; lauf <= tok; lauf++)
	{
	  if (strcasecmp(puffer[lauf],"[warning]") == 0)
	    warn = 1;
	  else
	    {
	      if ((strcasecmp(puffer[lauf],"step") == 0) && (strcasecmp(puffer[lauf+1],"to") == 0))
		{
		  while ((lauf < tok) && (strcasecmp(puffer[lauf],"source-line") != 0))
		    {
		      lauf++;
		      if (lauf >= tok)
			{
			  WriteText(message,"Source line unknown !");
			  break;
			}
		    }
		  anz_read++;
		  aktl = atoi(puffer[lauf+1]);
		  XawListHighlight(listfile, aktl-1);
		  erfolg = TRUE;
		  fileend = FALSE;
		  strcpy(t,"Step to line ");
		  itoa(aktl,t1);
		  strcat(t,t1);
		  WriteText(message,t);
		}
	      if (strcasecmp(puffer[lauf],"processors") == 0)
		{
		  if (warn)
		    ScanOut(0,1);
		  l2 = lauf + 2;
		  strcpy(actproc,puffer[l2]);
		  l2++;
		  while (puffer[l2][0] != 'P')
		    {
		      if (puffer[l2][0] == '\0')
			{
			  pascan1(inbuf,TRUE);
			  l2 = 0;
			}
		      if (puffer[l2][0] != 'P')
			{
			  strcat(actproc,puffer[l2]);
			  l2++;
			}
		    }
		  lauf = tok;
		}
	    }
	}

      if ((strcasecmp(puffer[l2],"P*") == 0) || (strcasecmp(puffer[tok],"P*") == 0))
	{
	  erfolg = TRUE;
	  fertig = TRUE;
	  fileend = FALSE;
	}
      

      if ((strcasecmp(puffer[0],"P>") == 0) && (anz_read == 0))
	{
	  erfolg = TRUE;
	  fertig = TRUE;
	  fileend = TRUE;
	  break;
	}
    }

  while (!fertig)
    {
      pascan1(dummy,TRUE);
      if (strcasecmp(puffer[atokens],"P*") == 0)
	{
	  erfolg = TRUE;
	  fertig = TRUE;
	  fileend = FALSE;
	}
    }
  
  shown = (Position)(aktl * FONT_HT + FONT_OFF);
  XawViewportSetCoordinates(listvp, (Position)0, shown);
  ScanOut(1,2);
  ScanOut(1,1);
  WatchOut();
  DrawPes(strlen(actproc),actproc);

  if (fileend)
    {
      WriteText(message,"Arrived at the end of the source code - Another STEP, OVER or GO will rerun it !");
      XawViewportSetCoordinates(listvp, (Position)0, (Position)(lastlines * FONT_HT + FONT_OFF));
      XawListHighlight(listfile, lastlines-1);
      outst = (char *)realloc(outst,2);
      strcpy(outst,"\0");
      XClearWindow(MyDisplay, XtWindow(ctrlstruc));
    }
}

/*********************************************
 ** FUNCTION: StepOver                      **
 ** PURPOSE:  Callback function widget over **
 ** INPUTS:   Xt                            **
 ** RETURNS:  none                          **
 ********************************************/

void StepOver(w, client_data, call_data)
     Widget w;
     XtPointer client_data, call_data;
{
  Widget lf = (Widget) client_data;
  int i,j,k=0,l,lauf,l2;
  BOOLEAN erfolg, fertig, fileend;
  char t[50],t1[5];
  int anz_read = 0;
  Position shown;
  char dummy[BSIZE];
  int tok,warn=0;

  erfolg = FALSE;
  fertig = FALSE;
  fileend = FALSE;

  if (!WaitingForInput)
    paprint("STEP OVER\n");
  
  WriteText(message,"RUNNING");
  WriteOut(outst);

  ScanOut(1,2);
  
  while (!erfolg)
    {
      count = -1;
      k = 0;
      while ((count == -1) && (k < 5000))
	{
	  k++;
	  pascan1(inbuf,TRUE);
	}

      if (k < 5000)
	{
	  while ((strcasecmp(puffer[atokens],"P*") != 0) && (strcasecmp(puffer[atokens],"P>") != 0))
	    {
	      pascan1(dummy,TRUE);
	      if (count != -1)
		strcat(inbuf,dummy);
	    }
      
	}

      clearbuffer();
      tok = scanner(inbuf);

      if (debugon)
	fprintf(stderr,"INBUF(OVER) = %s\n",inbuf);

      if (k >= 5000)
	{
	  XtSetSensitive(step, FALSE);
	  XtSetSensitive(over, FALSE);
	  XtSetSensitive(gline,FALSE);
	  XtSetSensitive(listfile,FALSE);
	  XtSetSensitive(load,FALSE);
	  XtSetSensitive(topol,FALSE);
	  XtSetSensitive(addwatch, FALSE);
	  XtSetSensitive(delwatch, FALSE);
	  XtSetSensitive(go,FALSE);
	  XtSetSensitive(edit,FALSE);
	  XtSetSensitive(clearbp,FALSE);

	  ScanOut(1,2);
	  WaitingForInput = TRUE;

	  id2 = XtAppAddInput (ApplicationContext,re2,XtInputReadMask,ScanOut,12);
	  id1 = XtAppAddInput (ApplicationContext,re1,XtInputReadMask,StepOver,listfile);
	  id1son = TRUE;
	  id2son = TRUE;
	  return;
	}
      else
	WaitingForInput = FALSE;

      if ((strncasecmp(inbuf,"program is not executable",25) == 0) || (!fileok))
	{
	  WriteText(message,"ERROR - No correct source file loaded !!!");
	  return;
	}

      for (lauf = 0; lauf <= tok; lauf++)
	{
	  if (strcasecmp(puffer[lauf],"[warning]") == 0)
	    warn = 1;
	  else
	    {
	      if (strcasecmp(puffer[lauf],"P*") == 0)
		{
		  XtSetSensitive(step, TRUE);
		  XtSetSensitive(over, TRUE);
		  XtSetSensitive(gline,TRUE);
		  XtSetSensitive(listfile, TRUE);
		  XtSetSensitive(load, TRUE);
		  XtSetSensitive(topol, TRUE);	      
		  XtSetSensitive(addwatch, TRUE);
		  XtSetSensitive(delwatch, TRUE);
		  XtSetSensitive(go,TRUE);
		  XtSetSensitive(edit,TRUE);
		  XtSetSensitive(clearbp,TRUE);
		  erfolg = TRUE;
		  fertig = TRUE;
		  fileend = FALSE;
		  if (id1son)
		    {
		      XtRemoveInput(id1);
		      id1son = FALSE;
		    }
		  if (id2son)
		    {
		      XtRemoveInput(id2);
		      id2son = FALSE;
		    }
		}
	      if ((strcasecmp(puffer[lauf],"step") == 0) && (strcasecmp(puffer[lauf+1],"to") == 0))
		{
		  while (strcasecmp(puffer[lauf++],"source-line") != 0);
		  XtSetSensitive(step, TRUE);
		  XtSetSensitive(over, TRUE);
		  XtSetSensitive(gline,TRUE);
		  XtSetSensitive(listfile, TRUE);
		  XtSetSensitive(load, TRUE);
		  XtSetSensitive(topol, TRUE);	      
		  XtSetSensitive(addwatch, TRUE);
		  XtSetSensitive(delwatch, TRUE);
		  XtSetSensitive(go,TRUE);
		  XtSetSensitive(edit,TRUE);
		  XtSetSensitive(clearbp,TRUE);
		  anz_read++;
		  aktl = atoi(puffer[lauf]);
		  XawListHighlight(listfile, aktl-1);
		  erfolg = TRUE;
		  fertig = TRUE;
		  fileend = FALSE;
		  strcpy(t,"Step over to line ");
		  itoa(aktl,t1);
		  strcat(t,t1);
		  WriteText(message,t);
		  if (id1son)
		    {
		      XtRemoveInput(id1);
		      id1son = FALSE;
		    }
		  if (id2son)
		    {
		      XtRemoveInput(id2);
		      id2son = FALSE;
		    }		  
		}
	      if (strcasecmp(puffer[lauf],"processors") == 0)
		{
		  if (warn)
		    ScanOut(0,1);
		  l2 = lauf + 2;
		  strcpy(actproc,puffer[l2]);
		  l2++;
		  while (puffer[l2][0] != 'P')
		    {
		      if (puffer[l2][0] == '\0')
			{
			  pascan1(inbuf,TRUE);
			  l2 = 0;
			}
		      if (puffer[l2][0] != 'P')
			{
			  strcat(actproc,puffer[l2]);
			  l2++;
			}
		    }
		  lauf = tok;
		}
	    }
	}
      if (strcasecmp(puffer[tok],"P*") == 0)
	{
	  erfolg = TRUE;
	  fertig = TRUE;
	  fileend = FALSE;
	}
      
      
      if ((strcasecmp(puffer[0],"P>") == 0) && (anz_read == 0))
	{
	  erfolg = TRUE;
	  fertig = TRUE;
	  fileend = TRUE;
	  if (id1son)
	    {
	      XtRemoveInput(id1);
	      id1son = FALSE;
	    }
	  if (id2son)
	    {
	      XtRemoveInput(id2);
	      id2son = FALSE;
	    }		  
	  break;
	}
    }

  while (!fertig)
    {
      pascan1(dummy,TRUE);
      if (strcasecmp(puffer[atokens],"P*") == 0)
        fertig = TRUE;
    }
  
  shown = (Position)(aktl * FONT_HT + FONT_OFF);
  XawViewportSetCoordinates(listvp, (Position)0, shown);
  ScanOut(1,2);
  ScanOut(1,1);
  WatchOut();
  DrawPes(strlen(actproc),actproc);

  if (fileend)
    {
      XtSetSensitive(step, TRUE);
      XtSetSensitive(over, TRUE);
      XtSetSensitive(gline,TRUE);
      XtSetSensitive(listfile, TRUE);
      XtSetSensitive(load, TRUE);
      XtSetSensitive(topol, TRUE);	      
      XtSetSensitive(addwatch, TRUE);
      XtSetSensitive(delwatch, TRUE);
      XtSetSensitive(go,TRUE);
      XtSetSensitive(edit,TRUE);
      XtSetSensitive(clearbp,TRUE);
      WriteText(message,"Arrived at the end of the source code - Another STEP or OVER will rerun it !");
      XawViewportSetCoordinates(listvp, (Position)0, (Position)(lastlines * FONT_HT + FONT_OFF));
      XawListHighlight(listfile, lastlines-1);
      
      outst = (char *)realloc(outst,2);
      strcpy(outst,"\0");
      XClearWindow(MyDisplay, XtWindow(ctrlstruc));
    }
}

/**********************************************
 ** FUNCTION: GotoLine                       **
 ** PURPOSE:  Callback function widget gline **
 ** INPUTS:   Xt                             **
 ** RETURNS:  none                           **
 *********************************************/

void GotoLine(w, client_data, call_data )
     Widget w;
     XtPointer client_data, call_data;
{
  Position x,y;
  Dimension width, height;

  if (!fileok)
    {
      WriteText(message,"ERROR - No source file loaded !!!");
      return;
    }
  
  XtVaGetValues( ctrlbutt, XtNwidth, &width, XtNheight, &height, NULL);
  XtTranslateCoords (ctrlbutt, (Position) width/2, (Position) height/2, &x, &y);
  XtVaSetValues( glpopup, XtNx, x, XtNy, y, NULL);
  XtSetSensitive (gline, FALSE);
  XtPopup(glpopup, XtGrabNonexclusive);
}

/**********************************************
** FUNCTION: GlDone                          **
** PURPOSE:  Callback function widget gldone **
** INPUTS:   Xt                              **
** RETURNS:  none                            **
**********************************************/

void GlDone(w, client_data, call_data )
     Widget w;
     XtPointer client_data, call_data;
{
  char temp[50],t1[50];
  BOOLEAN erfolg, fertig, running, error;
  int i,j,l2;
  int po,k;
  char *s;
  Position shown;
  int tok,warn=0;
  char dummy[BSIZE];
  int fehler = 0;

  erfolg = FALSE;
  fertig = FALSE;
  running = FALSE;
  error = FALSE;

  XtPopdown(glpopup);
  XtSetSensitive( gline, TRUE);
  
  XtVaGetValues( gldialog, XtNstring, &s, NULL);

  if (s[0] == '\0')
    {
      WriteText(message,"No line number entered !");
      return;
    }

  po = atoi(s);

  strcpy(temp,"GO TO ");
  itoa(po,t1);
  strcat(temp,t1);
  strcat(temp,"\n");

  if (!WaitingForInput)
    paprint(temp);

  WriteText(message,"RUNNING");

  XtSetSensitive(step, FALSE);
  XtSetSensitive(over, FALSE);
  XtSetSensitive(gline,FALSE);
  XtSetSensitive(go,FALSE);
  XtSetSensitive(listfile,FALSE);
  XtSetSensitive(load,FALSE);
  XtSetSensitive(topol,FALSE);
  XtSetSensitive(addwatch, FALSE);
  XtSetSensitive(delwatch, FALSE);
  XtSetSensitive(edit, FALSE);
  XtSetSensitive(clearbp,FALSE);

  fehler = ScanOut(1,2);

  while (!fertig)
    {
      while (!erfolg)
	{
	  if (!fehler)
	    pascan2(inbuf,TRUE);

	  for (i = 0; i <= atokens; i++)
	    if ((strcasecmp(puffer[i],"[error]") == 0) || (fehler))
	      {
		WriteText(message,"No move possible to this line number !!!");
		XtSetSensitive(step, TRUE);
		XtSetSensitive(over, TRUE);
		XtSetSensitive(go,TRUE);
		XtSetSensitive(listfile, TRUE);
		XtSetSensitive(load, TRUE);
		XtSetSensitive(topol, TRUE);            
		XtSetSensitive(addwatch, TRUE);
		XtSetSensitive(delwatch, TRUE);
		XtSetSensitive(edit, TRUE);
		XtSetSensitive(gline,TRUE);
		XtSetSensitive(clearbp,TRUE);
		pascan1(inbuf,TRUE);
		return;
	      }

	  ScanOut(0,2);
	  ScanOut(0,1);

	  count = -1;
	  k = 0;
          while ((count == -1) && (k < 5000))
            {
              pascan1(inbuf,TRUE);
              k++;
            }

	  if (k < 5000)
	    {
	      while ((strcasecmp(puffer[atokens],"P*") != 0) && (strcasecmp(puffer[atokens],"P>") != 0))
		{
		  pascan1(dummy,TRUE);
		  if (count != -1)
		    strcat(inbuf,dummy);
		}
	    }

	  clearbuffer();
	  tok = scanner(inbuf);

	  if (debugon)
	    fprintf(stderr,"INBUF(GLDONE) = %s\n",inbuf);

          if (k >= 5000)
            {
              count = -1;
              ScanOut(1,2);
              WaitingForInput = TRUE;

	      id2 = XtAppAddInput (ApplicationContext,re2,XtInputReadMask,ScanOut,12);
	      id1 = XtAppAddInput (ApplicationContext,re1,XtInputReadMask,GlDone,0);
	      id1gon = TRUE;
	      id2gon = TRUE;
	      return;
            }
          else
            WaitingForInput = FALSE;

          if ((strncasecmp(inbuf,"program is not executable",25) == 0) || (!fileok))
            {
              WriteText(message,"ERROR - No correct source file loaded !!!");
              XtSetSensitive(step, TRUE);
              XtSetSensitive(over, TRUE);
              XtSetSensitive(go,TRUE);
              XtSetSensitive(listfile, TRUE);
              XtSetSensitive(load, TRUE);
              XtSetSensitive(topol, TRUE);            
              XtSetSensitive(addwatch, TRUE);
              XtSetSensitive(delwatch, TRUE);
	      XtSetSensitive(gline,TRUE);
	      XtSetSensitive(edit, TRUE);
	      XtSetSensitive(clearbp,TRUE);
              erfolg = TRUE;
              fertig = TRUE;
	      running = FALSE;
	      error = TRUE;
              return;
	    }
	  for (i = 0; i <= tok; i++)
	    {
	      if (strcasecmp(puffer[i],"[warning]") == 0)
		warn = 1;
	      if (strcasecmp(puffer[i],"P>") == 0)
		{
		  if (id1gon)
		    {
		      XtRemoveInput(id1);
		      id1gon = FALSE;
		    }
		  if (id2gon)
		    {
		      XtRemoveInput(id2);
		      id2gon = FALSE;
		    }
		  i = tok+1;
		  erfolg = TRUE;
		  fertig = TRUE;
		  running = TRUE;
		}
	      if ((strcasecmp(puffer[i],"reached") == 0) ||
		  (strcasecmp(puffer[i],"interrupt") == 0) ||
		  (strcasecmp(puffer[i],"forced") == 0))
		{
		  erfolg = TRUE;
		  for (j = i; j <= tok; j++)
		    {
		      if (strcasecmp(puffer[j],"source-line") == 0)
			{
			  aktl = atoi(puffer[j+1]);
			  strcpy(temp,"Gone to line ");
			  strcat(temp,puffer[j+1]);
			  WriteText(message,temp);
			  if (id1gon)
			    {
			      XtRemoveInput(id1);
			      id1gon = FALSE;
			    }
			  if (id2gon)
			    {
			      XtRemoveInput(id2);
			      id2gon = FALSE;
			    }
			}
		      if (strcasecmp(puffer[j],"processors") == 0)
			{
			  if (warn)
			    ScanOut(0,1);
			  l2 = j + 2;
			  strcpy(actproc,puffer[l2]);
			  l2++;
			  while (puffer[l2][0] != 'P')
			    {
			      if (puffer[l2][0] == '\0')
				{
				  pascan1(inbuf,TRUE);
				  l2 = 0;
				}
			      if (puffer[l2][0] != 'P')
				{
				  strcat(actproc,puffer[l2]);
				  l2++;
				}
			    }
			}
		    }
		  if (strcasecmp(puffer[tok],"P>") == 0)
		    {
		      erfolg = TRUE;
		      fertig = TRUE;
		      running = TRUE;
		      error = FALSE;
		      i = tok+1;
		      j = tok+1;
		    }
		  if (strcasecmp(puffer[tok],"P*") == 0)
		    {
		      erfolg = TRUE;
		      fertig = TRUE;
		      running = FALSE;
		      error = FALSE;
		      i = tok+1;
		      j = tok+1;
		    }
		}
	    }
	}
      
      if (running)
	{
	  XtSetSensitive(step, TRUE);
	  XtSetSensitive(over, TRUE);
	  XtSetSensitive(go,TRUE);
	  XtSetSensitive(listfile, TRUE);
	  XtSetSensitive(load, TRUE);
	  XtSetSensitive(topol, TRUE);            
	  XtSetSensitive(addwatch, TRUE);
	  XtSetSensitive(delwatch, TRUE);
	  XtSetSensitive(edit, TRUE);
	  XtSetSensitive(gline,TRUE);
	  XtSetSensitive(clearbp,TRUE);
	  WriteText(message,"Arrived at the end of the source code !");
	  XawViewportSetCoordinates(listvp, (Position)0, (Position)(lastlines * FONT_HT + FONT_OFF));
	  XawListHighlight(listfile, lastlines-1);
	  aktl = 0;
	  outst = (char *)realloc(outst,2);
	  strcpy(outst,"\0");
	  WatchOut();
	  XClearWindow(MyDisplay, XtWindow(ctrlstruc));
	}      
      else
	if (erfolg)
	  {
	    XtSetSensitive(step, TRUE);
	    XtSetSensitive(over, TRUE);
	    XtSetSensitive(go,TRUE);
	    XtSetSensitive(listfile, TRUE);
	    XtSetSensitive(load, TRUE);
	    XtSetSensitive(topol, TRUE);            
	    XtSetSensitive(addwatch, TRUE);
	    XtSetSensitive(delwatch, TRUE);
	    XtSetSensitive(gline,TRUE);
	    XtSetSensitive(edit,TRUE);
	    XtSetSensitive(clearbp,TRUE);
	    shown = (Position)(aktl * FONT_HT + FONT_OFF);
	    XawViewportSetCoordinates(listvp, (Position)0, shown);
	    ScanOut(1,2);
	    ScanOut(1,1);
	    WatchOut();
	    DrawPes(strlen(actproc),actproc);
	    if (aktl > 0)
	      XawListHighlight(listfile, aktl-1);
	  }
    }
}

/********************************************************************************
 ** FUNCTION: InReturn                                                         **
 ** PURPOSE:  Action procedure, gets string from input widget and sends        **
 **           it to the simulator as a result of a Parallaxis READ instruction **
 ** INPUTS:   none                                                             **
 ** RETURNS:  none                                                             **
 *******************************************************************************/

void InReturn()
{
  char *s,*t;
  int k;

  XtVaGetValues(intext, XtNstring, &s, NULL);

  t = (char *)calloc(strlen(s)+2,sizeof(char));
  strcpy(t,s);
  strcat(t,"\n");
  paprint(t);

  outst = (char *)realloc(outst,strlen(outst)+strlen(t)+1);
  if (outst == NULL)
    {
      fprintf(stderr,"Couldn't get memory for OUTPUT-String !\n");
      exit(1);
    }
  strcat(outst,t);
  WriteOut(outst);
  
  strcpy(t,"\0");
  XtVaSetValues(intext, XtNstring, t, NULL);
  XtSetSensitive(ctrlbutt, TRUE);

  if (!free(t))
    fprintf(stderr,"Couldn't free T (InReturn) !\n");
}

/**************************************************
 ** FUNCTION: StopEx                             **
 ** PURPOSE:  Callback function widget stopex    **
 ** INPUTS:   Xt                                 **
 ** RETURNS:  none                               **
 *************************************************/

void StopEx(w, client_data, call_data)
     Widget w;
     XtPointer client_data, call_data;
{
  char t[100],t1[10];

  strcpy(t,"kill -2 ");
  itoa(pid,t1);
  strcat(t,t1);
  system(t);
  if (id1on)
    {
      XtRemoveInput(id1);
      id1on = FALSE;
      if (id2on)
	{
	  XtRemoveInput(id2);
	  id2on = FALSE;
	}
      Go(listfile);
    }
  if (id1gon)
    {
      XtRemoveInput(id1);
      id1gon = FALSE;
      if (id2gon)
	{
	  XtRemoveInput(id2);
	  id2gon = FALSE;
	}
      GlDone();
    }
  if (id1son)
    {
      XtRemoveInput(id1);
      id1son = FALSE;
      if (id2son)
	{
	  XtRemoveInput(id2);
	  id2son = FALSE;
	}
      StepOver(listfile);
    }
}  

/**************************************************
 ** FUNCTION: ClearBp                            **
 ** PURPOSE:  Callback function widget clearbp   **
 ** INPUTS:   Xt                                 **
 ** RETURNS:  none                               **
 *************************************************/

void ClearBp(w, client_data, call_data)
     Widget w;
     XtPointer client_data, call_data;
{
  int i,aline;
  char t[100],t1[10];

  for (i = 0; i < lastlines; i++)
    {
      if (bpoint[i] == 1)
	{
	  bpoint[i] = 0;
	  itoa(i,t1);
	  strcpy(t,"BREAKPOINT -");
	  strcat(t,t1);
	  strcat(t,"\n");
	  paprint(t);

	  aline = CheckForLine();

	  filebuf[aline-1][0] = ' ';
	}
    }
  DrawListFile(lastlines);
  XawListUnhighlight(listfile);
  if ((aktl > 0) && (aktl <= lines))
    XawListHighlight(listfile,aktl-1);
}

/**************************************************
 ** FUNCTION: AddWatch                           **
 ** PURPOSE:  Callback function widget addwatch  **
 ** INPUTS:   Xt                                 **
 ** RETURNS:  none                               **
 *************************************************/

void AddWatch(w, client_data, call_data)
     Widget w;
     XtPointer client_data, call_data;
{
  char *s = "\0";
  Position x,y;
  Dimension width, height;

  if (!fileok)
    {
      WriteText(message,"ERROR - No correct source file loaded !!!");
      return;
    }

  XtVaGetValues( ctrlbutt, XtNwidth, &width, XtNheight, &height, NULL);
  XtTranslateCoords (ctrlbutt, (Position) width/4, (Position) height/2, &x, &y);
  XtVaSetValues( wpopup, XtNx, x, XtNy, y, NULL);
  XtVaSetValues( wdialog, XtNstring, s, NULL); 
  XtSetSensitive (addwatch, FALSE);
  XtPopup(wpopup, XtGrabNonexclusive);
}

/*********************************************
** FUNCTION: WDone                          **
** PURPOSE:  Callback function widget wdone **
** INPUTS:   Xt                             **
** RETURNS:  none                           **
*********************************************/

void WDone(w, client_data, call_data)
     Widget w;
     XtPointer client_data, call_data;
{
  char *s1,*s2;
  char temp[500000],t1[100];
  BOOLEAN fertig = FALSE;
  int i,tok;

  XtPopdown(wpopup);
  XtSetSensitive( addwatch, TRUE);
  
  XtVaGetValues(wdialog, XtNstring, &s1, NULL);
  
  if (s1[0] == '\0')
    {
      WriteText(message,"No value entered, nothing will be displayed !");
      return;
    }

  ScanOut(1,2);
  ScanOut(1,1);
  
  strcpy(t1,"EXAMINE ");
  strcat(t1,s1);
  strcat(t1,"\n");
  paprint(t1);

  XtVaGetValues(petext, XtNstring, &s2, NULL);
  strcpy(temp,s2);

  strcpy(inbuf,"\0");
  strcpy(puffer[0],"\0");
  
  while (!fertig)
    {
      pascan1(inbuf,TRUE);
      tok = atokens;
      pascan2(inbuf,FALSE);
      strcat(temp,inbuf);

      for (i = 0; i <= tok; i++)
	if (((strcasecmp(puffer[i],"P*") == 0) || (strcasecmp(puffer[i],"P>") == 0)) ||
	    (strcasecmp(puffer[i],"[error]") == 0))
	  {
	    fertig = TRUE;
	    strcpy(watches[wpoints++],s1);      
	    i = tok+1;
	  }

    }

  CheckFor13(temp);
  strcat(temp,"---------------------------------------------------------------------------------\n");
  XtVaSetValues(petext, XtNstring, temp, NULL);
}

/*************************************************
 ** FUNCTION: DelWatch                          **
 ** PURPOSE:  Callback function widget delwatch **
 ** INPUTS:   Xt                                **
 ** RETURNS:  none                              **
 ************************************************/

void DelWatch(w, client_data, call_data)
     Widget w;
     XtPointer client_data, call_data;
{
  if (wpoints > 0)
    {
      strcpy(watches[--wpoints],"\0");
      WatchOut();
    }
  else
    {
      WriteText(message,"ERROR - There are no watch points defined !!!");
    }
}

/***************************************************
 ** FUNCTION: ShowHelp                            **
 ** PURPOSE:  Callback function widget helpbutton **
 ** INPUTS:   Xt                                  **
 ** RETURNS:  none                                **
 **************************************************/

void ShowHelp(w, client_data, call_data)
     Widget w;
     XtPointer client_data, call_data;
{
  Widget ctrl = (Widget) client_data;
  Position x,y;
  Dimension width, height;
  char htext[4000];
  strcpy(htext, "     --------------------------------------------------------------------------------------\n");
  strcat(htext, "     MISC:\n");
  strcat(htext, "     --------------------------------------------------------------------------------------\n");
  strcat(htext, "     Quit          - Exits the PGUI program.\n");
  strcat(htext, "     Help          - Shows this help page.\n");
  strcat(htext, "     --------------------------------------------------------------------------------------\n");
  strcat(htext, "     FILE COMMANDS:\n");
  strcat(htext, "     --------------------------------------------------------------------------------------\n");
  strcat(htext, "     Load file     - Opens a file selector box to choose a new Parallaxis source file.\n");
  strcat(htext, "     Edit file     - Opens the selected editor with the current Parallaxis work file.\n");
  strcat(htext, "     Choose editor - Opens a dialog box to enter your preffered text editor.\n");
  strcat(htext, "     --------------------------------------------------------------------------------------\n");
  strcat(htext, "     DEBUG COMMANDS:\n");
  strcat(htext, "     --------------------------------------------------------------------------------------\n");
  strcat(htext, "     Go            - Executes the PARZ-command GO to interprete the whole Parallaxis\n");
  strcat(htext, "                     program. If there are breakpoints set, execution will stop there.\n");
  strcat(htext, "     Step          - Executes the PARZ-command GO STEP to interprete just one Parallaxis\n");
  strcat(htext, "                     source line. Parallaxis procedures will be entered.\n");
  strcat(htext, "     Over          - Executes the PARZ-command GO OVER to interprete just one Parallaxis\n");
  strcat(htext, "                     source line. Parallaxis procedures will not be entered, they are\n");
  strcat(htext, "                     executed in one step.\n");
  strcat(htext, "     Goto line     - Executes the PARZ-command GO TO LINE to interprete the Parallaxis file\n");
  strcat(htext, "                     until the given source line is reached. The source line had to be\n");
  strcat(htext, "                     specified in the popup window. Exit by clicking the Done button or by\n");
  strcat(htext, "                     pressing Return.\n");
  strcat(htext, "     Add watch     - Allows you to watch valid Parallaxis-variables or -expressions.\n");
  strcat(htext, "                     The variables are collected in a list and updated after every\n");
  strcat(htext, "                     STEP or OVER command.\n"); 
  strcat(htext, "                     After choosing this button, a popup appears where you can enter the\n");
  strcat(htext, "                     Parallaxis expression. Exit by clicking on DONE or by pressing the\n");
  strcat(htext, "                     RETURN key.\n");
  strcat(htext, "     Delete watch  - Deletes the last item in the list of the Parallaxis variables.\n");
  strcat(htext, "                     Then this item will no longer be shown in the appropriate window.\n");
  strcat(htext, "     Display       - A popup allows you to enter the information needed for drawing the\n");
  strcat(htext, "                     processor topology.\n");
  strcat(htext, "                     In the first line you can enter the Parallaxis expression you whish to\n");
  strcat(htext, "                     show as colors and the range for the colormap.\n");
  strcat(htext, "                     If there is nothing entered, no colors will be displayed\n");
  strcat(htext, "                     In the second line you can decide whether the PE-labels should be drawn\n");
  strcat(htext, "                     or not. The Button toggles this label when clicked on.\n");
  strcat(htext, "                     In the third line you can choose whether the first PE in a grid topology\n");
  strcat(htext, "                     should be drawn in the upper or lower left corner.\n");
  strcat(htext, "                     In the fourth line there are three Buttons to exit the popup.\n");
  strcat(htext, "                     Clicking on DISPLAY AS GRID will draw the PEs as a grid, clicking\n");
  strcat(htext, "                     on DISPLAY AS CIRCLE will draw them as a circle and clicking on\n");
  strcat(htext, "                     DISPLAY NOTHING will draw nothing at all.\n");
  strcat(htext, "     ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++\n");
  strcat(htext, "     BREAKPOINTS:\n");
  strcat(htext, "     Set or remove breakpoints just by clicking on the appropriate source line.\n");
  strcat(htext, "     If a breakpoint is set on a line, then a 'B'-character left from the line\n");
  strcat(htext, "     number will show this.\n");
  strcat(htext, "     If a line with a 'B'-character is clicked, this character vanishes to reflect that\n");
  strcat(htext, "     this breakpoint is no longer set.\n");

  XtVaGetValues( ctrl, XtNwidth, &width, XtNheight, &height, NULL);
  XtTranslateCoords( ctrl, (Position) 0, (Position) height/2, &x, &y);
  XtVaSetValues( helppopup, XtNx, x, XtNy, y, NULL);
  XtVaSetValues( helptext, XtNstring, htext, NULL);
  XtPopup(helppopup, XtGrabNonexclusive); 
}

/*************************************************
 ** FUNCTION: HelpDone                          **
 ** PURPOSE:  Callback function widget helpdone **
 ** INPUTS:   Xt                                **
 ** RETURNS:  none                              **
 ************************************************/

void HelpDone(w, client_data, call_data)
     Widget w;
     XtPointer client_data, call_data;
{
  XtPopdown(helppopup);
}

/**************************************************
** FUNCTION: main                                **
** PURPOSE:  main program                        **
** INPUTS:   int argc, char **argv               **
** RETURNS:  none                                **
**************************************************/

void main(argc,argv)
     int argc;
     char **argv;
{
  int i;                                                    /* Count variable */
  BOOLEAN ok1 = FALSE, ok2 = FALSE, ok3 = FALSE;            /* Booleans for checking the correct  */
                                                            /* connection to the PARZ interpreter */
  char *s;

  if (!(s = getenv("EDITOR")))
    strcpy(editorstring,"xedit ");                          /* Default text editor */
  else
    {
      strcpy(editorstring,s);                               /* Editor from ENV-variable EDITOR */
      strcat(editorstring," ");
    }

  if (argc > 1)            
    if ((strcasecmp(argv[1],"debug") == 0) ||
        (strcasecmp(argv[1],"-debug") == 0) ||
        (strcasecmp(argv[1],"-d") == 0))
      debugon = TRUE;                                       /* Debug-Mode on */
  
  CreateAll(argc,argv);                                     /* Create all widgets */
  
  CreatePipe();                                             /* Create the three pipes */
  
  CreateFisWindow();                                        /* Create the file select window */
  
  while (!ok3)                                              /* wait until PARZ-Interpreter is ready */
    {
      pascan1(inbuf,TRUE);
      
      if (debugon)
	if (strlen(inbuf) > 0)
	  fprintf(stderr,"INBUF(START) = %s\n",inbuf);

      for (i = 0; i <= atokens; i++)
	{
	  if (strcasecmp(puffer[i],"PARZ") == 0)            /* Has PARZ been connected ? */
	    ok1 = TRUE;
	  if ((strcasecmp(puffer[i],"2.04,") == 0)
	      || (strcasecmp(puffer[i],"2.07,") == 0)
	      || (strcasecmp(puffer[i],"2.08,") == 0))      /* Is it the right version ? */
	    ok2 = TRUE;
	  if (strcasecmp(puffer[i],"P>") == 0)              /* Has the whole message been read ? */
	    ok3 = TRUE;
	}
    }

  clearbuffer();
  
  /* ok1 == FALSE : The PARZ interpreter is not connectied */
  if (!ok1)
    WriteText(message,"Couldn't connect the PARZ-Interpreter !!! Please check your settings and restart !");
  
  /* ok2 == FALSE : Wrong version of the interpreter */
  if (!ok2)
    WriteText(message,"This is not a correct Parallaxis Version (2.04 / 2.07 / 2.08) !!!");
  
  /* Adding callbacks */
  
  XtAddCallback(goodbye, XtNcallback, Quit, 0);
  XtAddCallback(load, XtNcallback, PopupFisWindow, ctrlbutt);
  XtAddCallback(editor,XtNcallback, Editor, ctrlbutt);
  XtAddCallback(done, XtNcallback, DialogDone, 0);
  XtAddCallback(edit, XtNcallback, Edit, 0);
  XtAddCallback(go, XtNcallback, Go, listfile);
  XtAddCallback(listfile, XtNcallback, Listfile, listfile);
  XtAddCallback(step, XtNcallback, GoStep, listfile);
  XtAddCallback(over, XtNcallback, StepOver, listfile);
  XtAddCallback(gline, XtNcallback, GotoLine, listfile);
  XtAddCallback(gldone, XtNcallback, GlDone, 0);
  XtAddCallback(addwatch, XtNcallback, AddWatch, 0);
  XtAddCallback(wdone, XtNcallback, WDone, 0);
  XtAddCallback(delwatch, XtNcallback, DelWatch, 0);
  XtAddCallback(helpbutton, XtNcallback, ShowHelp, ctrlbutt);
  XtAddCallback(helpdone, XtNcallback, HelpDone, 0);
  XtAddCallback(topol, XtNcallback, Topology, 0);
  XtAddCallback(tlabel, XtNcallback, TLabel, 0);
  XtAddCallback(tstart, XtNcallback, TStart, 0);
  XtAddCallback(tcircle, XtNcallback, TCircle, 0);
  XtAddCallback(tgrid, XtNcallback, TGrid, 0);
  XtAddCallback(tnothing, XtNcallback, TNothing, 0);
  XtAddCallback(stopex, XtNcallback, StopEx, 0);
  XtAddCallback(clearbp, XtNcallback, ClearBp, 0);
  
  /* Starting Main Loop */
  
  XtRealizeWidget(toplevel);
  XtAppMainLoop(ApplicationContext);
}
