/*****************************************************************************
  Project: PARZ - Parallel Intermediate Code Debugger/Interpreter
  ----------------------------------------------------------------------------
  Release      : 1
  Project Part : Simulation
  Filename     : do_3fu.c       
  SCCS-Path    : /tmp_mnt/user/sembach/parz/v2/SCCS/s.do_3fu.c
  Version      : 1.3 
  Last changed : 9/27/93 at 14:04:48        
  Author       : Frank Sembach
  Creation date: Aug. 92
  ----------------------------------------------------------------------------
  Description  : Funktionen zur Simulation der PARZ-Befehle

******************************************************************************
***      (C) COPYRIGHT University of Stuttgart - All Right Reserved        ***
*****************************************************************************/

static char sccs_id[] = "@(#)do_3fu.c	1.3  9/27/93 PARZ - Simulation (Frank Sembach)";


#include "parzdefs.h"
#include "rundefs.h"
#include "komdefs.h"
#include "y_tab.h"
#include "externs.h"
#include "runexts.h"
#include "toup.h"
#include "xgraph.h"

extern FILE *yyerfp;            /* Ausgabedatei fuer Fehlermeldungen */

extern char *wert_string();     /* aus Datei print_fu.c */

AKTBLOCK *temp_a_block;         /* Zeiger auf Aktivierungsblock,
                                   der durch CALL erzeugt wurde und
                                   von PROC vervollstaendigt werden muss */

/**************************************************************************
 ***                      Funktion hole_string
 ***
 *** Erzeugt aus dem Speicherbereich ab '*it' im Allozierungsblock '*bl'
 *** einen String.
 *** vlen     : '1' fuer skalaren, 'pe_anz' fuer vektoriellen Speicher
 *** *null_da : Rueckgabe TRUE, falls 'termS' gefunden wurde, sonst FALSE 
 *** Das Ergebnis ist ein Zeiger auf einen mit calloc() allozierten String.
 ***
 **************************************************************************/

unsigned char *hole_string(it, bl, vlen, null_da)
ITEM *it;
VARBLOCK *bl;
int vlen, *null_da;
{ register int count;
  register int max = (bl->bis - ((it - bl->vars) / vlen + bl->von) + 1);
  register unsigned char *ergstr, *sp;

  if (!(ergstr = (unsigned char *)calloc((size_t)(max + 1), (size_t)sizeof(char))))
  { runerr(texte[4]); return NULL; }
  *null_da = TRUE;
  for (sp = ergstr, count = max; 1; it += vlen, sp++, count--)
  { if (!count || !(it->datentyp & CHA_ERLAUBT) || ITEM_typ(*it) != CHA)
    { *sp = '\0';
      *null_da = FALSE;
      break;
    }
    *sp = it->inhalt.c_val;
    if (!*sp) break;
  }
  return ergstr;
}

/**************************************************************************
 ***                   Funktionen do_txy_strcmp
 ***
 *** fuehren Stringvergleichsbefehle 'verg := STRCMP vc1 vc2' aus:
 *** t = s : Skalarbefehl, t = v : Vektorbefehl
 *** x,y bezeichnen Argumentarten:
 ***     c   : CHAR-Array
 ***     str : konstante Zeichenkette
 *** verg wird -1, 0 bzw 1 fuer vc1 <, = bzw. > vc2
 *** Fehlermeldung, wenn kein Unterschied gefunden wurde und 
 *** mindestens ein String nicht durch 'termS' abgeschlossen ist
 ***
 **************************************************************************/

int do_scc_strcmp(bef)
STAT *bef;
{ register unsigned char *st1, *st2;
  register ITEM *ziel, *st1it, *st2it;
  register VARBLOCK *bl1, *bl2;
  register long erg;
  int terminated1, terminated2;

  protokoll('b', NULL);
  ziel = (zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp))->itps[0];
  if (err) return;
  st1it = item_ptr(&STAT_vc1(*bef), &dummybs, disp)->itps[0];
  if (err) return;
  bl1 = dummybs->blocks[0];
  st1 = hole_string(st1it, bl1, 1, &terminated1);
  if (err) return;
  st2it = item_ptr(&STAT_vc2(*bef), &dummybs, disp)->itps[0];
  if (err) return;
  bl2 = dummybs->blocks[0];
  st2 = hole_string(st2it, bl2, 1, &terminated2);
  if (err) return;
  if (terminated1 && terminated2)
  { erg = sign(strcmp(st1, st2)); }
  else
  { register int l1 = strlen(st1);
    register int l2 = strlen(st2);

    erg = sign(strncmp(st1, st2, (size_t)(l1 <= l2 && !terminated1) ? l1 : l2));
    if (erg == 0)
    { runerr(texte[146]); return; }
  }
  set_i_item(ziel, erg, 0);
  free(st1); free(st2);
}

int do_scstr_strcmp(bef)
STAT *bef;
{ register unsigned char *st1, *st2;
  register ITEM *ziel, *st1it;
  register VARBLOCK *bl;
  register long erg;
  int terminated;

  protokoll('b', NULL);
  ziel = (zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp))->itps[0];
  if (err) return;
  st1it = item_ptr(&STAT_vc1(*bef), &dummybs, disp)->itps[0];
  if (err) return;
  bl = dummybs->blocks[0];
  st1 = hole_string(st1it, bl, 1, &terminated);
  if (err) return;
  st2 = (unsigned char *)ARG_con_wert(STAT_vc2(*bef)).inhalt.s_val;
  if (terminated)
  { erg = sign(strcmp(st1, st2)); }
  else
  { erg = sign(strncmp(st1, st2, (size_t)strlen(st1)));
    if (erg == 0)
    { runerr(texte[146]); return; }
  }
  set_i_item(ziel, erg, 0);
  free(st1);
}

int do_sstrc_strcmp(bef)
STAT *bef;
{ register unsigned char *st1, *st2;
  register ITEM *ziel, *st2it;
  register VARBLOCK *bl;
  register long erg;
  int terminated;

  protokoll('b', NULL);
  ziel = (zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp))->itps[0];
  if (err) return;
  st2it = item_ptr(&STAT_vc2(*bef), &dummybs, disp)->itps[0];
  if (err) return;
  bl = dummybs->blocks[0];
  st2 = hole_string(st2it, bl, 1, &terminated);
  if (err) return;
  st1 = (unsigned char *)ARG_con_wert(STAT_vc1(*bef)).inhalt.s_val;
  if (terminated)
  { erg = sign(strcmp(st1, st2)); }
  else
  { erg = sign(strncmp(st1, st2, (size_t)strlen(st2)));
    if (erg == 0)
    { runerr(texte[146]); return; }
  }
  set_i_item(ziel, erg, 0);
  free(st2);
}

int do_sstrstr_strcmp(bef)
STAT *bef;
{ register ITEM *ziel = (zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp))->itps[0];

  if (err) return;
  protokoll('b', NULL);
  set_i_item(ziel, (long) sign(strcmp(ARG_con_wert(STAT_vc1(*bef)).inhalt.s_val,
                                      ARG_con_wert(STAT_vc2(*bef)).inhalt.s_val)), 0);
}

int do_vcc_strcmp(bef)
STAT *bef;
{ register unsigned char *st1, *st2;
  register ITEM *ziel, *st1it, *st2it;
  register VARBLOCK **bl1, **bl2;
  register ITPTRS *st1pt, *st2pt;
  VBLOCKS *bl1pt, *bl2pt;
  register long erg;
  register int vlen1, vlen2;
  int terminated1, terminated2;

  protokoll('b', NULL);
  zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp);
  if (err) return;
  st1pt = item_ptr(&STAT_vc1(*bef), &bl1pt, disp);
  if (err) return;
  vlen1 = (st1pt->n_it_func == gl_item) ? 1 : pe_anz;
  bl1 = &bl1pt->blocks[0];
  st2pt = item_ptr(&STAT_vc2(*bef), &bl2pt, disp);
  if (err) return;
  vlen2 = (st2pt->n_it_func == gl_item) ? 1 : pe_anz;
  bl2 = &bl2pt->blocks[0];
  vec_bef = TRUE;
  for (pec = 0; pec < last_akt_pe; pec++, bl1 += bl1pt->inc, bl2 += bl2pt->inc)
  { ziel = (* zielits->n_it_func)(zielits);
    st1it = (* st1pt->n_it_func)(st1pt);
    st2it = (* st2pt->n_it_func)(st2pt);
    if (aktive_pes[pec] == '1')
    { st1 = hole_string(st1it, *bl1, vlen1, &terminated1);
      if (err) return;
      st2 = hole_string(st2it, *bl2, vlen2, &terminated2);
      if (err) return;
      if (terminated1 && terminated2)
      { erg = sign(strcmp(st1, st2)); }
      else
      { register int l1 = strlen(st1);
        register int l2 = strlen(st2);

        erg = sign(strncmp(st1, st2, (size_t)(l1 <= l2 && !terminated1) ? l1 : l2));
        if (erg == 0)
        { runerr(texte[146]); return; }
      }
      set_i_item(ziel, erg, 0);
      free(st1); free(st2);
    }
  }
}

int do_vcstr_strcmp(bef)
STAT *bef;
{ register unsigned char *st1, *st2;
  register ITEM *ziel, *st1it;
  register VARBLOCK **bl1;
  register ITPTRS *st1pt;
  VBLOCKS *bl1pt;
  register long erg;
  register int vlen1;
  int terminated;

  protokoll('b', NULL);
  zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp);
  if (err) return;
  st1pt = item_ptr(&STAT_vc1(*bef), &bl1pt, disp);
  if (err) return;
  vlen1 = (st1pt->n_it_func == gl_item) ? 1 : pe_anz;
  bl1 = &bl1pt->blocks[0];
  st2 = (unsigned char *)ARG_con_wert(STAT_vc2(*bef)).inhalt.s_val;
  vec_bef = TRUE;
  for (pec = 0; pec < last_akt_pe; pec++, bl1 += bl1pt->inc)
  { ziel = (* zielits->n_it_func)(zielits);
    st1it = (* st1pt->n_it_func)(st1pt);
    if (aktive_pes[pec] == '1')
    { st1 = hole_string(st1it, *bl1, vlen1, &terminated);
      if (err) return;
      if (terminated)
      { erg = sign(strcmp(st1, st2)); }
      else
      { erg = sign(strncmp(st1, st2, (size_t)strlen(st1)));
        if (erg == 0)
        { runerr(texte[146]); return; }
      }
      set_i_item(ziel, erg, 0);
      free(st1);
    }
  }
}

int do_vstrc_strcmp(bef)
STAT *bef;
{ register unsigned char *st1, *st2;
  register ITEM *ziel, *st2it;
  register VARBLOCK **bl2;
  register ITPTRS *st2pt;
  VBLOCKS *bl2pt;
  register long erg;
  register int vlen2;
  int terminated;

  protokoll('b', NULL);
  zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp);
  if (err) return;
  st1 = (unsigned char *)ARG_con_wert(STAT_vc1(*bef)).inhalt.s_val;
  st2pt = item_ptr(&STAT_vc2(*bef), &bl2pt, disp);
  if (err) return;
  vlen2 = (st2pt->n_it_func == gl_item) ? 1 : pe_anz;
  bl2 = &bl2pt->blocks[0];
  vec_bef = TRUE;
  for (pec = 0; pec < last_akt_pe; pec++, bl2 += bl2pt->inc)
  { ziel = (* zielits->n_it_func)(zielits);
    st2it = (* st2pt->n_it_func)(st2pt);
    if (aktive_pes[pec] == '1')
    { st2 = hole_string(st2it, *bl2, vlen2, &terminated);
      if (err) return;
      if (terminated)
      { erg = sign(strcmp(st1, st2)); }
      else
      { erg = sign(strncmp(st1, st2, (size_t)strlen(st2)));
        if (erg == 0)
        { runerr(texte[146]); return; }
      }
      set_i_item(ziel, erg, 0);
      free(st2);
    }
  }
}

int do_vstrstr_strcmp(bef)
STAT *bef;
{ register ITEM *ziel;
  register long erg;

  protokoll('B', aktive_pes);
  zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp);
  if (err) return;
  erg = sign(strcmp(ARG_con_wert(STAT_vc1(*bef)).inhalt.s_val,
                    ARG_con_wert(STAT_vc2(*bef)).inhalt.s_val));
  vec_bef = TRUE;
  for (pec = 0; pec < last_akt_pe; pec++)
  { ziel = (* zielits->n_it_func)(zielits);
    if (aktive_pes[pec] == '1')
    { set_i_item(ziel, erg, 0); }
  }
}

/**************************************************************************
 ***                  Funktion add_connection
 ***
 *** erzeugt die Verbindung von 'von_pe' 'von_port' nach 'nach_pe' 'nach_port',
 *** falls 'von_pe' und 'nach_pe' korrekte PE-Nummern sind.
 ***
 **************************************************************************/

add_connection(von_pe, von_port, nach_pe, nach_port)
long von_pe, nach_pe;
int von_port, nach_port;
{ register int num;
  register PORT *outp, **neuarr;

  if (!(von_pe < 1l || von_pe > (long)pe_anz ||
        nach_pe < 1l || nach_pe > (long)pe_anz))
  { outp = &PNR((int)von_pe, von_port);
    num = outp->out_count++;
    if (!(neuarr = (PORT **)(outp->zielarr ?
                             realloc(outp->zielarr, (size_t)((num + 1) *
                                                             sizeof(PORT *))) :
                             calloc((size_t)(num + 1), (size_t)sizeof(PORT *)))))
    { runerr(texte[62]); return; }
    neuarr[num] = &PNR((int)nach_pe, nach_port);
    outp->zielarr = neuarr;
  }
}

/**************************************************************************
 ***                      Funktion do_connect
 ***
 *** fuehrt Befehl 'CONNECT out_port TO in_port AT vc1' aus
 ***
 **************************************************************************/

int do_connect(bef)
STAT *bef;
{ register ITEM *penr, *outp, *inp;
  register ITPTRS *penrits;
  register int outnum, innum;
  register long outn, inn;

  protokoll('T', aktive_pes);
  if (reducing)
  { runerr(texte[114]); return; }
  inp = item_ptr(&STAT_in_port(*bef), &dummybs, disp)->itps[0];
  if (err) return;
  ini_typ_test(inp->datentyp, INT, 91, 283);
  innum = (int)(inn = inp->inhalt.i_val);
  if (inn < 1l || inn > (long)port_anz)
  { runerr(texte[64], innum); return; }
  outp = item_ptr(&STAT_out_port(*bef), &dummybs, disp)->itps[0];
  if (err) return;
  ini_typ_test(outp->datentyp, INT, 91, 283);
  outnum = (int)(outn = outp->inhalt.i_val);
  if (outn < 1l || outn > (long)port_anz)
  { runerr(texte[64], outnum); return; }
  penrits = item_ptr(&STAT_vc1(*bef), &dummybs, disp);
  if (err) return;
  vec_bef = TRUE;
  for (pec = 0; pec < last_akt_pe; pec++)
  { penr = (* penrits->n_it_func)(penrits);
    if (aktive_pes[pec] == '1')
    { ini_typ_test(penr->datentyp, INT, 91, 283);
      add_connection((long)(pec + 1), outnum, penr->inhalt.i_val, innum);
      if (err) return;
    }
  }
}

/**************************************************************************
 ***                      Funktion do_biconnect
 ***
 *** fuehrt Befehl 'BICONNECT out_port TO in_port AT vc1' aus
 ***
 **************************************************************************/

int do_biconnect(bef)
STAT *bef;
{ register ITEM *penr, *outp, *inp;
  register ITPTRS *penrits;
  register int outnum, innum;
  register long outn, inn;

  protokoll('T', aktive_pes);
  if (reducing)
  { runerr(texte[114]); return; }
  inp = item_ptr(&STAT_in_port(*bef), &dummybs, disp)->itps[0];
  if (err) return;
  ini_typ_test(inp->datentyp, INT, 91, 283);
  innum = (int)(inn = inp->inhalt.i_val);
  if (inn < 1l || inn > (long)port_anz)
  { runerr(texte[64], innum); return; }
  outp = item_ptr(&STAT_out_port(*bef), &dummybs, disp)->itps[0];
  if (err) return;
  ini_typ_test(outp->datentyp, INT, 91, 283);
  outnum = (int)(outn = outp->inhalt.i_val);
  if (outn < 1l || outn > (long)port_anz)
  { runerr(texte[64], outnum); return; }
  penrits = item_ptr(&STAT_vc1(*bef), &dummybs, disp);
  if (err) return;
  vec_bef = TRUE;
  for (pec = 0; pec < last_akt_pe; pec++)
  { penr = (* penrits->n_it_func)(penrits);
    if (aktive_pes[pec] == '1')
    { ini_typ_test(penr->datentyp, INT, 91, 283);
      add_connection((long)(pec + 1), outnum, penr->inhalt.i_val, innum);
      if (err) return;
      add_connection(penr->inhalt.i_val, innum, (long)(pec + 1), outnum);
      if (err) return;
    }
  }
}

/**************************************************************************
 ***                      Funktion do_disconnect
 ***
 *** fuehrt Befehl 'DISCONNECT' aus
 ***
 **************************************************************************/

int do_disconnect(bef)
STAT *bef;
{ register PORT *poptr;
  register int poc;

  protokoll('T', aktive_pes);
  if (reducing)
  { runerr(texte[114]); return; }
  vec_bef = TRUE;
  if (pe_anz * port_anz)
  { for (poptr = portarray, pec = 0; pec < pe_anz; pec++)
    { if (aktive_pes[pec] == '1')
      { for (poc = 0; poc < port_anz; poc++, poptr++)
        { poptr->out_count = 0;
          if (poptr->zielarr) free(poptr->zielarr);
          poptr->zielarr = NULL;
        }
      }
      else poptr += port_anz;
    }
  }
}

/**************************************************************************
 ***                      Funktion do_one_disconnect
 ***
 *** fuehrt Befehl 'DISCONNECT out_port' aus
 ***
 **************************************************************************/

int do_one_disconnect(bef)
STAT *bef;
{ register ITEM *outp;
  register int outnum;
  register long outn;
  register PORT *poptr;

  protokoll('T', aktive_pes);
  if (reducing)
  { runerr(texte[114]); return; }
  outp = item_ptr(&STAT_out_port(*bef), &dummybs, disp)->itps[0];
  if (err) return;
  ini_typ_test(outp->datentyp, INT, 91, 283);
  outnum = (int)(outn = outp->inhalt.i_val);
  if (outn < 1l || outn > (long)port_anz)
  { runerr(texte[64], outnum); return; }
  for (poptr = &PNR(1, outnum), pec = 0; pec < pe_anz; pec++, poptr += port_anz)
  { if (aktive_pes[pec] == '1')
    { poptr->out_count = 0;
      free(poptr->zielarr);
      poptr->zielarr = NULL;
    }
  }
}

/********************** Vergleichsfunktionen ***************************/

int b_eq(b1,b2)
int b1, b2;
{ return (b1 == b2);
}

int c_eq(c1,c2)
unsigned char c1, c2;
{ return (c1 == c2);
}

int i_eq(i1,i2)
long i1, i2;
{ return (i1 == i2);
}

int r_eq(r1,r2)
float r1, r2;
{ return (r1 == r2);
}

int b_ne(b1,b2)
int b1, b2;
{ return (b1 != b2);
}

int c_ne(c1,c2)
unsigned char c1, c2;
{ return (c1 != c2);
}

int i_ne(i1,i2)
long i1, i2;
{ return (i1 != i2);
}

int r_ne(r1,r2)
float r1, r2;
{ return (r1 != r2);
}

int b_lt(b1,b2)
int b1, b2;
{ return (b1 < b2);
}

int c_lt(c1,c2)
unsigned char c1, c2;
{ return (c1 < c2);
}

int i_lt(i1,i2)
long i1, i2;
{ return (i1 < i2);
}

int r_lt(r1,r2)
float r1, r2;
{ return (r1 < r2);
}

int b_le(b1,b2)
int b1, b2;
{ return (b1 <= b2);
}

int c_le(c1,c2)
unsigned char c1, c2;
{ return (c1 <= c2);
}

int i_le(i1,i2)
long i1, i2;
{ return (i1 <= i2);
}

int r_le(r1,r2)
float r1, r2;
{ return (r1 <= r2);
}

int b_gt(b1,b2)
int b1, b2;
{ return (b1 > b2);
}

int c_gt(c1,c2)
unsigned char c1, c2;
{ return (c1 > c2);
}

int i_gt(i1,i2)
long i1, i2;
{ return (i1 > i2);
}

int r_gt(r1,r2)
float r1, r2;
{ return (r1 > r2);
}

int b_ge(b1,b2)
int b1, b2;
{ return (b1 >= b2);
}

int c_ge(c1,c2)
unsigned char c1, c2;
{ return (c1 >= c2);
}

int i_ge(i1,i2)
long i1, i2;
{ return (i1 >= i2);
}

int r_ge(r1,r2)
float r1, r2;
{ return (r1 >= r2);
}

int (* v_fs[REAL + 1][GE - EQ + 1])() =
{ {b_eq, b_ne, b_lt, b_le, b_gt, b_ge},
  {c_eq, c_ne, c_lt, c_le, c_gt, c_ge},
  {i_eq, i_ne, i_lt, i_le, i_gt, i_ge},
  {r_eq, r_ne, r_lt, r_le, r_gt, r_ge}
};

/***********************************************************************/

/**************************************************************************
 ***                   Funktion s_if_while_call
 ***
 *** gemeinsamer Teil der Funktionen die die Skalarbefehle
 *** 'if vc1 call spr_ziel' und 'while vc1 call spr_ziel' ausfuehren
 *** *bef : auszufuehrende Programmzeile
 *** ret  : Ruecksprungadresse fuer CALL (pc fuer while, pc+1 fuer if)
 ***
 **************************************************************************/

s_if_while_call(bef, ret)
STAT *bef;
int ret;
{ register ITEM *vergl;

  vergl = item_ptr(&STAT_vc1(*bef), &dummybs, disp)->itps[0];
  if (err) return;
  ini_typ_test(vergl->datentyp, BOOL, 91, 283);
  if (vergl->inhalt.b_val)
  { temp_a_block = new_aktblock(ret, NULL, FALSE);
    called = TRUE;
    call_count++;
    pc = STAT_spr_ziel(*bef) - 1;
  }
}

/**************************************************************************
 ***            Funktion do_s_ifcall
 ***
 *** fuehrt Skalarbefehl 'if vc1 call spr_ziel' aus
 ***
 **************************************************************************/

int do_s_ifcall(bef)
STAT *bef;
{ protokoll('i', NULL);
  s_if_while_call(bef, pc + 1);
}

/**************************************************************************
 ***           Funktion do_s_whilecall
 ***
 *** fuehrt Skalarbefehl 'while vc1 call spr_ziel' aus
 ***
 **************************************************************************/

int do_s_whilecall(bef)
STAT *bef;
{ protokoll('w', NULL);
  s_if_while_call(bef, pc);
}

/**************************************************************************
 ***                 Funktion s2_if_while_call
 ***
 *** gemeinsamer Teil der Funktionen die die Skalarbefehle
 *** 'if vc1 rel_tok vc2 call spr_ziel' und
 *** 'while vc1 rel_tok vc2 call spr_ziel' ausfuehren
 *** *bef : auszufuehrende Programmzeile
 *** ret  : Ruecksprungadresse fuer CALL (pc fuer while, pc+1 fuer if)
 *** verwendet zum Vergleich die Funktionen in v_fs[][]
 ***
 **************************************************************************/

s2_if_while_call(bef, ret)
STAT *bef;
int ret;
{ register ITEM *v1, *v2;
  register int erg;
  register TYP t = ARG_valtyp(STAT_vc1(*bef));

  v1 = item_ptr(&STAT_vc1(*bef), &dummybs, disp)->itps[0];
  if (err) return;
  ini_typ_test(v1->datentyp, t, 101, 284);
  v2 = item_ptr(&STAT_vc2(*bef), &dummybs, disp)->itps[0];
  if (err) return;
  ini_typ_test(v2->datentyp, t, 102, 285);
  switch (t)
  { case BOOL :
      erg = (* v_fs[BOOL][STAT_rel_tok(*bef) - EQ])(v1->inhalt.b_val, v2->inhalt.b_val);
      break;
    case CHA  :
      erg = (* v_fs[CHA][STAT_rel_tok(*bef) - EQ])(v1->inhalt.c_val, v2->inhalt.c_val);
      break;
    case INT  :
      erg = (* v_fs[INT][STAT_rel_tok(*bef) - EQ])(v1->inhalt.i_val, v2->inhalt.i_val);
      break;
    case REAL :
      erg = (* v_fs[REAL][STAT_rel_tok(*bef) - EQ])(v1->inhalt.r_val, v2->inhalt.r_val);
      break;
    default : bug("do_3fu.c/s2_if_while_call : falscher Typ");
  }
  if (erg)
  { temp_a_block = new_aktblock(ret, NULL, FALSE);
    called = TRUE;
    call_count++;
    pc = STAT_spr_ziel(*bef) - 1;
  }
}

/**************************************************************************
 ***                Funktion do_s2_ifcall
 ***
 *** fuehrt Skalarbefehl 'if vc1 rel_tok vc2 call spr_ziel' aus
 ***
 **************************************************************************/

int do_s2_ifcall(bef)
STAT *bef;
{ protokoll('i', NULL);
  s2_if_while_call(bef, pc + 1);
}

/**************************************************************************
 ***                Funktion do_s2_whilecall
 ***
 *** fuehrt Skalarbefehl 'while vc1 rel_tok vc2 call spr_ziel' aus
 ***
 **************************************************************************/

int do_s2_whilecall(bef)
STAT *bef;
{ protokoll('w', NULL);
  s2_if_while_call(bef, pc);
}

/**************************************************************************
 ***                Funktion v_if_while_call
 ***
 *** gemeinsamer Teil der Funktionen die die Vektorbefehle
 *** 'if vc1 call spr_ziel' und 'while vc1 call spr_ziel' ausfuehren
 *** *bef : auszufuehrende Programmzeile
 *** ret  : Ruecksprungadresse fuer CALL (pc fuer while, pc+1 fuer if)
 ***
 **************************************************************************/

v_if_while_call(bef, ret)
STAT *bef;
int ret;
{ register ITEM *vergl;
  register char *akts;
  register int sprungflag = FALSE;
  register ITPTRS *verglits;

  verglits = item_ptr(&STAT_vc1(*bef), &dummybs, disp);
  if (err) return;
  if (!(akts = calloc((size_t)(pe_anz + 1), (size_t)sizeof(char))))
  { runerr(texte[121]); return; }
  vec_bef = TRUE;
  for (pec = 0; pec < pe_anz; pec++)
  { vergl = (* verglits->n_it_func)(verglits);
    if (aktive_pes[pec] == '1')
    { ini_typ_test(vergl->datentyp, BOOL, 91, 283);
      akts[pec] = vergl->inhalt.b_val ? '1' : '0';
      sprungflag = sprungflag || vergl->inhalt.b_val;
    }
    else akts[pec] = '0';
  }
  if (sprungflag)
  { temp_a_block = new_aktblock(ret, aktive_pes, FALSE);
    aktive_pes = akts; last_akt_pe = last(aktive_pes,&anz_akt_pes);
    called = TRUE;
    call_count++;
    pc = STAT_spr_ziel(*bef) - 1;
  }
  else free(akts);
}

/**************************************************************************
 ***            Funktion do_v_ifcall
 ***
 *** fuehrt Vectorbefehl 'if vc1 call spr_ziel' aus
 ***
 **************************************************************************/

int do_v_ifcall(bef)
STAT *bef;
{ protokoll('I', aktive_pes);
  v_if_while_call(bef, pc + 1);
}

/**************************************************************************
 ***            Funktion do_v_whilecall
 ***
 *** fuehrt Vectorbefehl 'while vc1 call spr_ziel' aus
 ***
 **************************************************************************/

int do_v_whilecall(bef)
STAT *bef;
{ protokoll('W', aktive_pes);
  v_if_while_call(bef, pc);
}

/**************************************************************************
 ***               Funktion v2_if_while_call
 ***
 *** gemeinsamer Teil der Funktionen die die Vektorbefehle
 *** 'if vc1 rel_tok vc2 call spr_ziel' und
 *** 'while vc1 rel_tok vc2 call spr_ziel' ausfuehren
 *** *bef : auszufuehrende Programmzeile
 *** ret  : Ruecksprungadresse fuer CALL (pc fuer while, pc+1 fuer if)
 *** verwendet zum Vergleich die Funktionen in v_fs[][]
 ***
 **************************************************************************/

v2_if_while_call(bef, ret)
STAT *bef;
int ret;
{ register ITEM *v1, *v2;
  register int erg;
  register int sprungflag = FALSE;
  register char *akts;
  register ITPTRS *v1its, *v2its;
  register TYP t = ARG_valtyp(STAT_vc1(*bef));

  v1its = item_ptr(&STAT_vc1(*bef), &dummybs, disp);
  if (err) return;
  v2its = item_ptr(&STAT_vc2(*bef), &dummybs, disp);
  if (err) return;
  if (!(akts = calloc((size_t)(pe_anz + 1), (size_t)sizeof(char))))
  { runerr(texte[121]); return; }
  vec_bef = TRUE;
  for (pec = 0; pec < pe_anz; pec++)
  { v1 = (* v1its->n_it_func)(v1its);
    v2 = (* v2its->n_it_func)(v2its);
    if (aktive_pes[pec] == '1')
    { ini_typ_test(v1->datentyp, t, 101, 284);
      ini_typ_test(v2->datentyp, t, 102, 285);
      switch (t)
      { case BOOL :
          erg = (* v_fs[BOOL][STAT_rel_tok(*bef) - EQ])(v1->inhalt.b_val, v2->inhalt.b_val);
          break;
        case CHA  :
          erg = (* v_fs[CHA][STAT_rel_tok(*bef) - EQ])(v1->inhalt.c_val, v2->inhalt.c_val);
          break;
        case INT  :
          erg = (* v_fs[INT][STAT_rel_tok(*bef) - EQ])(v1->inhalt.i_val, v2->inhalt.i_val);
          break;
        case REAL :
          erg = (* v_fs[REAL][STAT_rel_tok(*bef) - EQ])(v1->inhalt.r_val, v2->inhalt.r_val);
          if (fperror) return;
          break;
        default : bug("do_3fu.c/v2_if_while_call : falscher Typ");
      }
      akts[pec] = erg ? '1' : '0';
      sprungflag = sprungflag || erg;
    }
    else akts[pec] = '0';
  }
  if (sprungflag)
  { temp_a_block = new_aktblock(ret, aktive_pes, FALSE);
    aktive_pes = akts; last_akt_pe = last(aktive_pes,&anz_akt_pes);
    called = TRUE;
    call_count++;
    pc = STAT_spr_ziel(*bef) - 1;
  }
  else free(akts);
}

/**************************************************************************
 ***                Funktion do_v2_ifcall
 ***
 *** fuehrt Vektorbefehl 'if vc1 rel_tok vc2 call spr_ziel' aus
 ***
 **************************************************************************/

int do_v2_ifcall(bef)
STAT *bef;
{ protokoll('I', aktive_pes);
  v2_if_while_call(bef, pc + 1);
}

/**************************************************************************
 ***                Funktion do_v2_whilecall
 ***
 *** fuehrt Vektorbefehl 'while vc1 rel_tok vc2 call spr_ziel' aus
 ***
 **************************************************************************/

int do_v2_whilecall(bef)
STAT *bef;
{ protokoll('W', aktive_pes);
  v2_if_while_call(bef, pc);
}

/**************************************************************************
 ***           Funktion do_ifgoto
 ***
 *** fuehrt Befehl 'if vc1 goto spr_ziel' aus
 ***
 **************************************************************************/

int do_ifgoto(bef)
STAT *bef;
{ register ITEM *vergl = item_ptr(&STAT_vc1(*bef), &dummybs, disp)->itps[0];

  protokoll('i', NULL);
  if (err) return;
  ini_typ_test(vergl->datentyp, BOOL, 91, 283);
  if (vergl->inhalt.b_val)
  { pc = STAT_spr_ziel(*bef) - 1; }
}

/**************************************************************************
 ***            Funktion do_2_ifgoto
 ***
 *** fuehrt Befehl 'if vc1 rel_tok vc2 goto spr_ziel' aus
 *** verwendet zum Vergleich die Funktionen in v_fs[][]
 ***
 **************************************************************************/

int do_2_ifgoto(bef)
STAT *bef;
{ register ITEM *v1, *v2;
  register int erg;
  register TYP t = ARG_valtyp(STAT_vc1(*bef));

  protokoll('i', NULL);
  v1 = item_ptr(&STAT_vc1(*bef), &dummybs, disp)->itps[0];
  if (err) return;
  ini_typ_test(v1->datentyp, t, 101, 284);
  v2 = item_ptr(&STAT_vc2(*bef), &dummybs, disp)->itps[0];
  if (err) return;
  ini_typ_test(v2->datentyp, t, 102, 285);
  switch (t)
  { case BOOL :
      erg = (* v_fs[BOOL][STAT_rel_tok(*bef) - EQ])(v1->inhalt.b_val, v2->inhalt.b_val);
      break;
    case CHA  :
      erg = (* v_fs[CHA][STAT_rel_tok(*bef) - EQ])(v1->inhalt.c_val, v2->inhalt.c_val);
      break;
    case INT  :
      erg = (* v_fs[INT][STAT_rel_tok(*bef) - EQ])(v1->inhalt.i_val, v2->inhalt.i_val);
      break;
    case REAL :
      erg = (* v_fs[REAL][STAT_rel_tok(*bef) - EQ])(v1->inhalt.r_val, v2->inhalt.r_val);
      break;
    default : bug("do_3fu.c/do_2_ifgoto : falscher Typ");
  }
  if (erg)
  { pc = STAT_spr_ziel(*bef) - 1; }
}

/**************************************************************************
 ***       Funktion do_goto
 ***
 *** fuehrt Befehl 'goto spr_ziel' aus
 ***
 **************************************************************************/

int do_goto(bef)
STAT *bef;
{ protokoll('g', NULL);
  pc = STAT_spr_ziel(*bef) - 1;
}

/**************************************************************************
 ***                    Funktion liesstring
 ***
 *** liest aus 'f' einen String in neu allozierten Speicher
 *** fuehrende Zeichen <= ' ' werden ueberlesen
 *** String besteht nur aus Zeichen > ' '
 *** ist 'termch_aendern' == TRUE, wird das erste nicht eingelesene Zeichen
 *** nach 'termCH' geschrieben
 *** 'liesstring' liefert NULL, falls wg. EOF kein String eingelesen
 *** wurde, sonst Zeiger auf gelesenen String.
 *** wird EOF gelesen, erhaelt 'termCH' gegebenenfalls den Wert 'termS'
 ***
 **************************************************************************/

char *liesstring(f, termch_aendern)
FILE *f;
int termch_aendern;
{ register char *s, *sp, *neu;
  register int ch;
  register int anf = FALSE;
  register int len = 256;

  do
  { ch = getc(f);
#ifdef MAC
    if (ch == EOF)
    { if (termch_aendern) set_c_item(&s_spez_vars[TERMCH_VAR], TERMS_CHAR, 0);
      if (f == stdin) zeile_fertig = TRUE;
      return NULL;
    }
    else if (quitted && ch == '\n' && f == stdin)
    { if (termch_aendern) set_c_item(&s_spez_vars[TERMCH_VAR], EOL_CHAR, 0);
      zeile_fertig = TRUE;
      return NULL;
    }
#else
    if (ch == EOF)
    { if (termch_aendern) set_c_item(&s_spez_vars[TERMCH_VAR], TERMS_CHAR, 0);
      if (ch == EOF && f == stdin) zeile_fertig = TRUE;
      return NULL;
    }
#endif
    else if (ch > ' ')
    { anf = TRUE; }
  } while (!anf);
  if (!(s = calloc((size_t)len, (size_t)sizeof(char))))
  { runerr(texte[80],36); return NULL; }
  sp = s;
  do 
  { *(sp++) = (char)ch;
    if (sp == s + len - 1)
    { *sp = '\0';
      neu = realloc(s, (size_t)((len += 10) * sizeof(char)));
      if (!neu)
      { ch = TERMS_CHAR;
        break;
      }
      else
      { s = neu;
        sp = s + strlen(s);
      }
    }
    ch = getc(f);
    if (ch == EOF)
    { anf = FALSE;
      ch = TERMS_CHAR;
    }
    else if (ch <= ' ')
    { anf = FALSE;
      ungetc(ch, f);
    }
  } while (anf);
  if (termch_aendern) set_c_item(&s_spez_vars[TERMCH_VAR], (unsigned char)ch, 0);
  if (f == stdin) zeile_fertig = FALSE;
  *sp = TERMS_CHAR;
  return s;
}

/**************************************************************************
 ***             Funktionen do_tx_read
 ***
 *** fuehren Lesebefehle 'READ verg' aus:
 *** t = s : Skalarbefehl, t = v : Vektorbefehl
 *** x bezeichnet Typ:
 ***    b : BOOLEAN
 ***    c : CHAR
 ***    i : INTEGER
 ***    r : REAL
 *** 'Done' kennzeichnet Erfolg/Nichterfolg der Leseoperation
 *** Ausser do_tc_read verwenden alle Funktionen 'liesstring()' und
 *** beschreiben 'termCH'
 ***
 **************************************************************************/

int do_sb_read(bef)
STAT *bef;
{ register char *s;
  register ITEM *ziel = (zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp))->itps[0];
  register char *sp;
  register int done_erg;

  protokoll('o', NULL);
  if (err) return;
  if (!(s = liesstring(runin, TRUE)))
  { done_erg = FALSE; }
  else
  { for (sp = s; *sp; *sp = mytoupper(*sp), sp++);
    if (strcmp(s, "TRUE") == 0)
    { done_erg = TRUE;
      set_b_item(ziel, TRUE, 0);
    }
    else if (strcmp(s, "FALSE") == 0)
    { done_erg = TRUE;
      set_b_item(ziel, FALSE, 0);
    }
    else
    { done_erg = FALSE; }
    free(s);
  }
  set_b_item(&s_spez_vars[DONE_VAR], done_erg, 0);
}

int do_sc_read(bef)
STAT *bef;
{ register ITEM *ziel = (zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp))->itps[0];
  register int inchar;

  if (err) return;
  protokoll('o', NULL);
  inchar = getc(runin);
  if (inchar == EOF)
  { set_b_item(&s_spez_vars[DONE_VAR], FALSE, 0);
    set_c_item(ziel, TERMS_CHAR, 0);
  }
  else
  { set_b_item(&s_spez_vars[DONE_VAR], TRUE, 0);
    set_c_item(ziel, (unsigned char)inchar, 0);
    if (runin == stdin)
    { zeile_fertig = (inchar == EOL_CHAR); }
  }
}

int do_si_read(bef)
STAT *bef;
{ register char *s;
  register ITEM *ziel = (zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp))->itps[0];
  long inzahl;
  register int done_erg;

  if (err) return;
  protokoll('o', NULL);
  if (!(s = liesstring(runin, TRUE)))
  { done_erg = FALSE; }
  else
  { if (!(sscanf(s, "%ld", &inzahl) == 1))
    { done_erg = FALSE; }
    else
    { done_erg = TRUE;
      set_i_item(ziel, inzahl, 0);
    }
    free(s);
  }
  set_b_item(&s_spez_vars[DONE_VAR], done_erg, 0);
}

int do_sr_read(bef)
STAT *bef;
{ register char *s;
  register ITEM *ziel = (zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp))->itps[0];
  float inzahl;
  register int done_erg;

  if (err) return;
  protokoll('o', NULL);
  if (!(s = liesstring(runin, TRUE)))
  { done_erg = FALSE; }
  else
  { if (!(sscanf(s, "%e", &inzahl) == 1))
    { done_erg = FALSE; }
    else
    { done_erg = TRUE;
      set_r_item(ziel, inzahl, 0);
    }
    free(s);
  }
  set_b_item(&s_spez_vars[DONE_VAR], done_erg, 0);
}

int do_vb_read(bef)
STAT *bef;
{ register char *s;
  register ITEM *ziel;
  register char *sp;
  register int done_erg = TRUE;

  protokoll('O', aktive_pes);
  zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp);
  if (err) return;
  for (pec = 0; pec < last_akt_pe; pec++)
  { ziel = (* zielits->n_it_func)(zielits);
    if (aktive_pes[pec] == '1')
    { if (!(s = liesstring(runin, TRUE)))
      { done_erg = FALSE; }
      else
      { for (sp = s; *sp; *sp = mytoupper(*sp), sp++);
        if (strcmp(s, "TRUE") == 0)
        { done_erg = TRUE;
          set_b_item(ziel, TRUE, 0);
        }
        else if (strcmp(s, "FALSE") == 0)
        { done_erg = TRUE;
          set_b_item(ziel, FALSE, 0);
        }
        else
        { done_erg = FALSE; }
        free(s);
      }
    }
  }
  set_b_item(&s_spez_vars[DONE_VAR], done_erg, 0);
}

int do_vc_read(bef)
STAT *bef;
{ register ITEM *ziel;
  register int inchar;
  register int done_erg = TRUE;

  protokoll('O', aktive_pes);
  zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp);
  if (err) return;
  for (pec = 0; pec < last_akt_pe; pec++)
  { ziel = (* zielits->n_it_func)(zielits);
    if (aktive_pes[pec] == '1')
    { inchar = getc(runin);
      if (inchar == EOF)
      { done_erg = FALSE;
        set_c_item(ziel, TERMS_CHAR, 0);
      }
      else
      { done_erg = TRUE;
        set_c_item(ziel, (unsigned char)inchar, 0);
        if (runin == stdin)
        { zeile_fertig = (inchar == EOL_CHAR); }
      } 
    }
  }
  set_b_item(&s_spez_vars[DONE_VAR], done_erg, 0);
}

int do_vi_read(bef)
STAT *bef;
{ register char *s;
  register ITEM *ziel;
  long inzahl;
  register int done_erg = TRUE;

  protokoll('O', aktive_pes);
  zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp);
  if (err) return;
  for (pec = 0; pec < last_akt_pe; pec++)
  { ziel = (* zielits->n_it_func)(zielits);
    if (aktive_pes[pec] == '1')
    { if (!(s = liesstring(runin, TRUE)))
      { done_erg = FALSE; }
      else
      { if (!(sscanf(s, "%ld", &inzahl) == 1))
        { done_erg = FALSE; }
        else
        { done_erg = TRUE;
          set_i_item(ziel, inzahl, 0);
        }
        free(s);
      }
    }
  }
  set_b_item(&s_spez_vars[DONE_VAR], done_erg, 0);
}

int do_vr_read(bef)
STAT *bef;
{ register char *s;
  register ITEM *ziel;
  float inzahl;
  register int done_erg = TRUE;

  protokoll('O', aktive_pes);
  zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp);
  if (err) return;
  for (pec = 0; pec < last_akt_pe; pec++)
  { ziel = (* zielits->n_it_func)(zielits);
    if (aktive_pes[pec] == '1')
    { if (!(s = liesstring(runin, TRUE)))
      { done_erg = FALSE; }
      else
      { if (!(sscanf(s, "%e", &inzahl) == 1))
        { done_erg = FALSE; }
        else
        { done_erg = TRUE;
          set_r_item(ziel, inzahl, 0);
        }
        free(s);
      }
    }
  }
  set_b_item(&s_spez_vars[DONE_VAR], done_erg, 0);
}

/**************************************************************************
 ***            Funktionen do_tread_string
 ***
 *** t = s : Skalarbefehl, t = v : Vektorbefehl
 *** fuehrt Befehl 'READ verg vc1' aus
 *** verwendet 'liesstring()', beschreibt 'termCH' und 'Done'
 *** verg muss Anfang eines CHAR-Arrays sein
 *** vc1 gibt Feldgroesse an
 *** Fehlermeldung, wenn das Feld zu kurz ist
 ***
 **************************************************************************/

int do_sread_string(bef)
STAT *bef;
{ register unsigned char *s;
  register ITEM *ziel;
  register int z, fertig;
  register unsigned char *sp;
  register ITEM *zaehl = item_ptr(&STAT_vc1(*bef), &dummybs, disp)->itps[0];
  register VARBLOCK *bl;
  register int done_erg;

  if (err) return;
  protokoll('o', NULL);
  ini_typ_test(zaehl->datentyp, INT, 91, 283);
  z = zaehl->inhalt.i_val;
  ziel = (zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp))->itps[0];
  if (err) return;
  bl = zielbs->blocks[0];
  if ((ziel - bl->vars) + bl->von + z - 1 > bl->bis)
  { runerr(texte[108]); return; }
  if (!(s = (unsigned char *)liesstring(runin, TRUE)))
  { done_erg = FALSE; }
  else
  { if (z < strlen(s)) s[z - 1] = TERMS_CHAR;
    for (sp = s, fertig = FALSE; !fertig; sp++, ziel++)
    { if (!(ziel->datentyp & CHA_ERLAUBT))
      { runerr(texte[109]); return; }
      set_c_item(ziel, *sp, 0);
      if (err) { free(s); return; }
      fertig = !(*sp);
    }
    done_erg = TRUE;
    free(s);
  }
  set_b_item(&s_spez_vars[DONE_VAR], done_erg, 0);
}

int do_vread_string(bef)
STAT *bef;
{ register unsigned char *s;
  register ITEM *ziel;
  register int z, fertig;
  register unsigned char *sp;
  register ITEM *zaehl = item_ptr(&STAT_vc1(*bef), &dummybs, disp)->itps[0];
  register VARBLOCK **blpt;
  register int done_erg = TRUE;

  protokoll('O', aktive_pes);
  if (err) return;
  ini_typ_test(zaehl->datentyp, INT, 91, 283);
  z = zaehl->inhalt.i_val;
  zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp);
  if (err) return;
  blpt = zielbs->blocks;
  for (pec = 0; pec < last_akt_pe; pec++, blpt += zielbs->inc)
  { ziel = (* zielits->n_it_func)(zielits);
    if (aktive_pes[pec] == '1')
    { register VARBLOCK *bl = *blpt;

      if ((ziel - bl->vars) / pe_anz + bl->von + z - 1 > bl->bis)
      { runerr(texte[108]); return; }
      if (!(s = (unsigned char *)liesstring(runin, TRUE)))
      { done_erg = FALSE; }
      else
      { if (z < strlen(s)) s[z - 1] = TERMS_CHAR;
        for (sp = s, fertig = FALSE; !fertig; sp++, ziel += pe_anz)
        { if (!(ziel->datentyp & CHA_ERLAUBT))
          { runerr(texte[109]); return; }
          set_c_item(ziel, *sp, 0);
          if (err) { free(s); return; }
          fertig = !(*sp);
        }
        done_erg = TRUE;
        free(s);
      }
    }
  }
  set_b_item(&s_spez_vars[DONE_VAR], done_erg, 0);
}

/**************************************************************************
 ***     Funktion do_t1_write
 ***
 *** t = s : Skalarbefehl, t = v : Vektorbefehl
 *** fuehrt Befehl 'WRITE vc1' aus
 ***
 **************************************************************************/

int do_s1_write(bef)
STAT *bef;
{ register ITEM *it = item_ptr(&STAT_vc1(*bef), &dummybs, disp)->itps[0];
  register TYP talt, t = ARG_valtyp(STAT_vc1(*bef));

  if (err) return;
  protokoll('o', NULL);
  ini_typ_test(it->datentyp, t, 91, 283);
  talt = it->datentyp;
  it->datentyp = CH_TYP(it->datentyp, t);
  if (t == CHA)
  { register char ch;

    putc((ch = (char)it->inhalt.c_val), runout);
    if (runout == stdout) no_linefeed = (ch != '\n');
  }
  else
  { register char *str = wert_string(it, t);

    if (str)
    { fputs(str, runout);
      if (runout == stdout) no_linefeed = (str[strlen(str) - 1] != '\n');
    }
  }
  it->datentyp = talt;
}

int do_v1_write(bef)
STAT *bef;
{ register ITEM *it;
  register ITPTRS *itps = item_ptr(&STAT_vc1(*bef), &dummybs, disp);
  register TYP talt, t = ARG_valtyp(STAT_vc1(*bef));
  register char ch;

  protokoll('O', aktive_pes);
  if (err) return;
  for (pec = 0; pec < last_akt_pe; pec++)
  { it = (* itps->n_it_func)(itps);
    if (aktive_pes[pec] == '1')
    { ini_typ_test(it->datentyp, t, 91, 283);
      talt = it->datentyp;
      it->datentyp = CH_TYP(it->datentyp, t);
      if (t == CHA)
      { putc((ch = (char)it->inhalt.c_val), runout);
      }
      else
      { register char *str = wert_string(it, t);

        if (str)
        { fputs(str, runout);
          putc('\n', runout);
        }
      }
      it->datentyp = talt;
    }
  }
  if (runout == stdout)
  { no_linefeed = (t == CHA) ? (ch != '\n') : FALSE; }
}

/**************************************************************************
 ***             Funktionen do_tx2_write
 ***
 *** t = s : Skalarbefehl, t = v : Vektorbefehl
 *** fuehren Schreibbefehle 'write vc1 vc2' aus:
 *** x bezeichnet Typ der Ausgabedaten vc1:
 ***    i : INTEGER                                                           
 ***    r : REAL
 ***    str : CHAR-Array
 *** vc2 bestimmt Ausgabelaenge (i,r : minimal; str : maximal)
 ***
 **************************************************************************/

int do_si2_write(bef)
STAT *bef;
{ register ITEM *it, *len;

  it = item_ptr(&STAT_vc1(*bef), &dummybs, disp)->itps[0];
  if (err) return;
  protokoll('o', NULL);
  ini_typ_test(it->datentyp, INT, 101, 284);
  len = item_ptr(&STAT_vc2(*bef), &dummybs, disp)->itps[0];
  if (err) return;
  ini_typ_test(len->datentyp, INT, 102, 285);
  fprintf(runout,"%*ld", (int)len->inhalt.i_val, it->inhalt.i_val);
  if (runout == stdout) no_linefeed = TRUE;
}

int do_sr2_write(bef)
STAT *bef;
{ register ITEM *it, *len;

  it = item_ptr(&STAT_vc1(*bef), &dummybs, disp)->itps[0];
  if (err) return;
  protokoll('o', NULL);
  ini_typ_test(it->datentyp, REAL, 101, 284);
  len = item_ptr(&STAT_vc2(*bef), &dummybs, disp)->itps[0];
  if (err) return;
  ini_typ_test(len->datentyp, INT, 102, 285);
  fprintf(runout,"%#.*e",
          (int)len->inhalt.i_val - ((it->inhalt.r_val < 0) ? 7 : 6),
          it->inhalt.r_val);
  if (runout == stdout) no_linefeed = TRUE;
}

int do_sstr2_write(bef)
STAT *bef;
{ register ITEM *it, *len;
  register unsigned char *ausgabe, *sptr;
  register int i;
  register VARBLOCK *bl;

  len = item_ptr(&STAT_vc2(*bef), &dummybs, disp)->itps[0];
  if (err) return;
  protokoll('o', NULL);
  ini_typ_test(len->datentyp, INT, 102, 285);
  it = item_ptr(&STAT_vc1(*bef), &dummybs, disp)->itps[0];
  if (err) return;
  bl = dummybs->blocks[0];
  if (!(ausgabe = (unsigned char *)calloc((size_t)((i = len->inhalt.i_val) + 1), 
                                          (size_t)sizeof(char))))
  { runerr(texte[4]); return; }
  if ((it - bl->vars) + bl->von + i - 1 > bl->bis)
  { runerr(texte[158]); free(ausgabe); return; }
  for (sptr = ausgabe; i; i--, sptr++, it++)
  { if (!(it->datentyp & CHA_ERLAUBT))
    { runerr(texte[159]); free(ausgabe); return; }
    ini_typ_test(it->datentyp, CHA, 101, 284);
    if (!(*sptr = it->inhalt.c_val)) break;
  }
  *sptr = '\0';
  fputs((char *)ausgabe, runout);
  if (runout == stdout) no_linefeed = (ausgabe[strlen(ausgabe) - 1] != '\n');
  free(ausgabe);
}

int do_vi2_write(bef)
STAT *bef;
{ register ITEM *it, *len;
  register int leng;
  register ITPTRS *itps = item_ptr(&STAT_vc1(*bef), &dummybs, disp);

  protokoll('O', aktive_pes);
  if (err) return;
  len = item_ptr(&STAT_vc2(*bef), &dummybs, disp)->itps[0];
  if (err) return;
  ini_typ_test(len->datentyp, INT, 102, 285);
  leng = (int)len->inhalt.i_val;
  for (pec = 0; pec < last_akt_pe; pec++)
  { it = (* itps->n_it_func)(itps);
    if (aktive_pes[pec] == '1')
    { ini_typ_test(it->datentyp, INT, 101, 284);
      fprintf(runout,"%*ld\n", leng, it->inhalt.i_val);
    }
  }
  no_linefeed = FALSE;
}

int do_vr2_write(bef)
STAT *bef;
{ register ITEM *it, *len;
  register int leng;
  register ITPTRS *itps = item_ptr(&STAT_vc1(*bef), &dummybs, disp);

  protokoll('O', aktive_pes);
  if (err) return;
  len = item_ptr(&STAT_vc2(*bef), &dummybs, disp)->itps[0];
  if (err) return;
  ini_typ_test(len->datentyp, INT, 102, 285);
  leng = (int)len->inhalt.i_val;
  for (pec = 0; pec < last_akt_pe; pec++)
  { it = (* itps->n_it_func)(itps);
    if (aktive_pes[pec] == '1')
    { ini_typ_test(it->datentyp, REAL, 101, 284);
      fprintf(runout,"%#.*e\n",
              (int)len->inhalt.i_val - ((it->inhalt.r_val < 0) ? 7 : 6),
              it->inhalt.r_val);
    }
  }
  no_linefeed = FALSE;
}

int do_vstr2_write(bef)
STAT *bef;
{ register ITEM *it;
  register ITEM *len = item_ptr(&STAT_vc2(*bef), &dummybs, disp)->itps[0];
  register unsigned char *ausgabe, *sptr;
  register int leng;
  register VARBLOCK *bl;
  register ITPTRS *itps;
  register VARBLOCK **blpt;

  protokoll('O', aktive_pes);
  if (err) return;
  ini_typ_test(len->datentyp, INT, 102, 285);
  itps = item_ptr(&STAT_vc1(*bef), &dummybs, disp);
  if (err) return;
  blpt = dummybs->blocks;
  if (!(ausgabe =
        (unsigned char *)calloc((size_t)((leng = len->inhalt.i_val) + 1), 
                                (size_t)sizeof(unsigned char))))
  { runerr(texte[4]); return; }
  for (pec = 0; pec < last_akt_pe; pec++, blpt += dummybs->inc)
  { it = (* itps->n_it_func)(itps);
    if (aktive_pes[pec] == '1')
    { register int i;
      register VARBLOCK *bl = *blpt;

      if ((it - bl->vars) / pe_anz + bl->von + leng - 1 > bl->bis)
      { runerr(texte[158]); free(ausgabe); return; }
      for (sptr = ausgabe, i = leng; i; i--, sptr++, it += pe_anz)
      { if (!(it->datentyp & CHA_ERLAUBT))
        { runerr(texte[159]); free(ausgabe); return; }
        ini_typ_test(it->datentyp, CHA, 101, 284);
        if (!(*sptr = it->inhalt.c_val)) break;
      }
      *sptr = '\0';
      fputs((char *)ausgabe, runout); putc('\n', runout);
    }
  }
  no_linefeed = FALSE;
  free(ausgabe);
}

/**************************************************************************
 ***         Funktion do_tr3_write
 ***
 *** t = s : Skalarbefehl, t = v : Vektorbefehl
 *** fuehrt Befehl 'write vc1 vc2 vc3' aus:
 *** vc1 : Ausgabedaten muessen REAL sein
 *** vc2 : minimale Ausgabelaenge
 *** vc3 : Nachkommastellen
 ***
 **************************************************************************/

int do_sr3_write(bef)
STAT *bef;
{ register ITEM *it, *len, *nach;

  it = item_ptr(&STAT_vc1(*bef), &dummybs, disp)->itps[0];
  if (err) return;
  protokoll('o', NULL);
  ini_typ_test(it->datentyp, REAL, 91, 283);
  len = item_ptr(&STAT_vc2(*bef), &dummybs, disp)->itps[0];
  if (err) return;
  ini_typ_test(len->datentyp, INT, 91, 283);
  nach = item_ptr(&STAT_vc3(*bef), &dummybs, disp)->itps[0];
  if (err) return;
  ini_typ_test(nach->datentyp, INT, 91, 283);
  fprintf(runout,"%#*.*f", (int)len->inhalt.i_val, (int)nach->inhalt.i_val, it->inhalt.r_val);
  if (runout == stdout) no_linefeed = TRUE;
}

int do_vr3_write(bef)
STAT *bef;
{ register ITEM *it, *len, *nach;
  register int l, n;
  register ITPTRS *itps = item_ptr(&STAT_vc1(*bef), &dummybs, disp);

  protokoll('O', aktive_pes);
  if (err) return;
  len = item_ptr(&STAT_vc2(*bef), &dummybs, disp)->itps[0];
  if (err) return;
  ini_typ_test(len->datentyp, INT, 91, 283);
  l = (int)len->inhalt.i_val;
  nach = item_ptr(&STAT_vc3(*bef), &dummybs, disp)->itps[0];
  if (err) return;
  ini_typ_test(nach->datentyp, INT, 91, 283);
  n = (int)nach->inhalt.i_val;
  for (pec = 0; pec < last_akt_pe; pec++)
  { it = (* itps->n_it_func)(itps);
    if (aktive_pes[pec] == '1')
    { ini_typ_test(it->datentyp, REAL, 91, 283);
      fprintf(runout,"%#*.*f\n", l, n, it->inhalt.r_val);
    }
  }
  no_linefeed = FALSE;
}

/**************************************************************************
 ***         Funktionen do_??_writeln
 ***
 *** entsprechen den Funktionen 'do_??_write',
 *** geben aber ein zusaetzliches 'EOL' aus
 ***    'do_v1_writeln' gibt nur bei 'char'-Argument ein
 ***    zusaetzliches 'EOL' aus. 
 ***
 **************************************************************************/

int do_s1_writeln(bef)
STAT *bef;
{ do_s1_write(bef);
  putc('\n', runout);
  if (runout == stdout) no_linefeed = FALSE;
}

int do_v1_writeln(bef)
STAT *bef;
{ do_v1_write(bef);
  if (ARG_typ(STAT_vc1(*bef)) == CHA)
  { putc('\n', runout);
    if (runout == stdout) no_linefeed = FALSE;
  }
}

int do_si2_writeln(bef)
STAT *bef;
{ do_si2_write(bef);
  putc('\n', runout);
  if (runout == stdout) no_linefeed = FALSE;
}

int do_sr2_writeln(bef)
STAT *bef;
{ do_sr2_write(bef);
  putc('\n', runout);
  if (runout == stdout) no_linefeed = FALSE;
}

int do_sstr2_writeln(bef)
STAT *bef;
{ do_sstr2_write(bef);
  putc('\n', runout);
  if (runout == stdout) no_linefeed = FALSE;
}

int do_sr3_writeln(bef)
STAT *bef;
{ do_sr3_write(bef);
  putc('\n', runout);
  if (runout == stdout) no_linefeed = FALSE;
}

/**************************************************************************
 ***     Funktion do_s1_draw
 ***
 *** fuehrt Befehl 'DRAW vc1' aus
 ***
 **************************************************************************/

int do_s1_draw(bef)
STAT *bef;
{ register ITEM *it = item_ptr(&STAT_vc1(*bef), &dummybs, disp)->itps[0];
  register TYP talt, t = ARG_valtyp(STAT_vc1(*bef));
  register int erg;

  if (err) return;
  protokoll('o', NULL);
  ini_typ_test(it->datentyp, t, 91, 283);
  talt = it->datentyp;
  it->datentyp = CH_TYP(it->datentyp, t);
  { register char *str = wert_string(it, t);

    if (str)
      { erg = DrawString(str);
	free(str);
      }
    else erg = FALSE;
  }
  it->datentyp = talt;
  set_b_item(&s_spez_vars[DONE_VAR], erg, 0);
}

/**************************************************************************
 ***             Funktionen do_sx2_draw
 ***
 *** fuehren graphische Schreibbefehle 'DRAW vc1 vc2' aus:
 *** x bezeichnet Typ der Ausgabedaten vc1:
 ***    i : INTEGER                                                           
 ***    r : REAL
 ***    str : CHAR-Array
 *** vc2 bestimmt Ausgabelaenge (i,r : minimal; str : maximal)
 ***
 **************************************************************************/

int do_si2_draw(bef)
STAT *bef;
{ register ITEM *it, *len;
  register int erg;

  it = item_ptr(&STAT_vc1(*bef), &dummybs, disp)->itps[0];
  if (err) return;
  protokoll('o', NULL);
  ini_typ_test(it->datentyp, INT, 101, 284);
  len = item_ptr(&STAT_vc2(*bef), &dummybs, disp)->itps[0];
  if (err) return;
  ini_typ_test(len->datentyp, INT, 102, 285);
  set_b_item(&s_spez_vars[DONE_VAR],
	     DrawInt((int)it->inhalt.i_val, (int)len->inhalt.i_val),
	     0);
}

int do_sr2_draw(bef)
STAT *bef;
{ register ITEM *it, *len;
  register int erg;

  it = item_ptr(&STAT_vc1(*bef), &dummybs, disp)->itps[0];
  if (err) return;
  protokoll('o', NULL);
  ini_typ_test(it->datentyp, REAL, 101, 284);
  len = item_ptr(&STAT_vc2(*bef), &dummybs, disp)->itps[0];
  if (err) return;
  ini_typ_test(len->datentyp, INT, 102, 285);
  set_b_item(&s_spez_vars[DONE_VAR],
	     DrawReal(it->inhalt.r_val, (int)len->inhalt.i_val),
	     0);
}

int do_sstr2_draw(bef)
STAT *bef;
{ register ITEM *it, *len;
  register unsigned char *ausgabe, *sptr;
  register int i, erg;
  register VARBLOCK *bl;

  len = item_ptr(&STAT_vc2(*bef), &dummybs, disp)->itps[0];
  if (err) return;
  protokoll('o', NULL);
  ini_typ_test(len->datentyp, INT, 102, 285);
  it = item_ptr(&STAT_vc1(*bef), &dummybs, disp)->itps[0];
  if (err) return;
  bl = dummybs->blocks[0];
  if (!(ausgabe = (unsigned char *)calloc((size_t)((i = len->inhalt.i_val) + 1), 
                                          (size_t)sizeof(char))))
  { runerr(texte[4]); return; }
  if ((it - bl->vars) + bl->von + i - 1 > bl->bis)
  { runerr(texte[158]); free(ausgabe); return; }
  for (sptr = ausgabe; i; i--, sptr++, it++)
  { if (!(it->datentyp & CHA_ERLAUBT))
    { runerr(texte[159]); free(ausgabe); return; }
    ini_typ_test(it->datentyp, CHA, 101, 284);
    if (!(*sptr = it->inhalt.c_val)) break;
  }
  *sptr = '\0';
  set_b_item(&s_spez_vars[DONE_VAR],
	     DrawString((char *)ausgabe),
	     0);
  free(ausgabe);
}

/**************************************************************************
 ***         Funktion do_sr3_draw
 ***
 *** fuehrt Befehl 'DRAW vc1 vc2 vc3' aus:
 *** vc1 : Ausgabedaten muessen REAL sein
 *** vc2 : minimale Ausgabelaenge
 *** vc3 : Nachkommastellen
 ***
 **************************************************************************/

int do_sr3_draw(bef)
STAT *bef;
{ register ITEM *it, *len, *nach;
  register int erg;

  it = item_ptr(&STAT_vc1(*bef), &dummybs, disp)->itps[0];
  if (err) return;
  protokoll('o', NULL);
  ini_typ_test(it->datentyp, REAL, 91, 283);
  len = item_ptr(&STAT_vc2(*bef), &dummybs, disp)->itps[0];
  if (err) return;
  ini_typ_test(len->datentyp, INT, 91, 283);
  nach = item_ptr(&STAT_vc3(*bef), &dummybs, disp)->itps[0];
  if (err) return;
  ini_typ_test(nach->datentyp, INT, 91, 283);
  set_b_item(&s_spez_vars[DONE_VAR],
	     DrawFixPt(it->inhalt.r_val,
		       (int)len->inhalt.i_val,
		       (int)nach->inhalt.i_val),
	     0);
}

/**************************************************************************
 ***                  Funktion do_load
 ***
 *** fuehrt Befehl 'LOAD verg WITH vc1' fuer pe_nr == 0 bzw.
 ***               'LOAD verg WITH vc1 PE pe_nr' aus
 ***
 **************************************************************************/

int do_load(bef)
STAT *bef;
{ register ITEM *quell, *ziel, *maxit;
  register VARBLOCK *bq;
  register TYP t = ARG_valtyp(STAT_verg(*bef));
  register long maxtr = (ITEM_typ(s_spez_vars[MAXTRANS_VAR]) == INT)
                        ? s_spez_vars[MAXTRANS_VAR].inhalt.i_val : (long)pe_anz;
  register long acttr = 0;
  register char *akt_pes;
  register int last_akt, anz_akt_save;

  if (reducing)
  { runerr(texte[114]); return; }
  zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp);
  if (err) return;
  quell = item_ptr(&STAT_vc1(*bef), &dummybs, disp)->itps[0];
  if (err) return;
  bq = dummybs->blocks[0];
  maxit = bq->vars - bq->von + bq->bis;
  anz_akt_save = anz_akt_pes;
  if (last_akt = STAT_pe_nr(*bef))         /* nur ein PE aktivieren */
  { akt_pes = dummy_aktive_pes;
    if ((akt_pes[last_akt - 1] = aktive_pes[last_akt - 1]) == '0')
    { runwarn(texte[231]); anz_akt_pes = 0;}
    else
    { anz_akt_pes = 1; }
  }
  else
  { akt_pes = aktive_pes; last_akt = last_akt_pe;
    vec_bef = TRUE;
  }
  protokoll('L', akt_pes);
  for (pec = 0; pec < last_akt && maxtr; pec++)
  { ziel = (* zielits->n_it_func)(zielits);
    if (akt_pes[pec] == '1')
    { if (quell > maxit)
      { runerr(texte[110]);
        dummy_aktive_pes[last_akt - 1] = '0';
        anz_akt_pes = anz_akt_save;
        return;
      }
      if (!(quell->datentyp & ERLAUBT(t)))
      { runerr(texte[84]);
        dummy_aktive_pes[last_akt - 1] = '0';
        anz_akt_pes = anz_akt_save;
        return;
      }
      set_item(ziel, quell, t);
      if (err || fperror)
      { dummy_aktive_pes[last_akt - 1] = '0';
        anz_akt_pes = anz_akt_save;
        return;
      }
      acttr++; maxtr--; quell++;
    }
  }
  anz_akt_pes = anz_akt_save;
  dummy_aktive_pes[last_akt - 1] = '0';
  s_spez_vars[MAXTRANS_VAR].datentyp = KEINER | INT_ERLAUBT;    /* MaxTrans wird uninitialisiert */
  set_i_item(&s_spez_vars[ACTTRANS_VAR], acttr, 0);
}

/**************************************************************************
 ***                  Funktion do_blockload
 ***
 *** fuehrt Befehl 'LOAD verg WITH vc1 AS dload' aus
 ***
 **************************************************************************/

int do_blockload(bef)
STAT *bef;
{ ITEM *quell, *ziel;
  register ITEM *maxit;
  register VARBLOCK *bq, *bz, **bzpt;
  register long maxtr = (ITEM_typ(s_spez_vars[MAXTRANS_VAR]) == INT)
                        ? s_spez_vars[MAXTRANS_VAR].inhalt.i_val : (long)pe_anz;
  register long acttr = 0;
  register int blocklen = STAT_dload(*bef)->typ_zahl[SUMM];

  protokoll('L', aktive_pes);
  if (reducing)
  { runerr(texte[114]); return; }
  zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp);
  if (err) return;
  bzpt = zielbs->blocks;
  quell = item_ptr(&STAT_vc1(*bef), &dummybs, disp)->itps[0];
  if (err) return;
  bq = dummybs->blocks[0];
  maxit = bq->vars - bq->von + bq->bis;
  vec_bef = TRUE;
  for (pec = 0; pec < last_akt_pe && maxtr; pec++, bzpt += zielbs->inc)
  { ziel = (* zielits->n_it_func)(zielits);
    if (aktive_pes[pec] == '1')
    { bz = *bzpt;
      if (quell > maxit - blocklen + 1)
      { runerr(texte[110]); return; }
      if ((ziel - bz->vars) / pe_anz + bz->von + blocklen - 1 > bz->bis)
      { runerr(texte[235]); return; }
      move_as_decl(STAT_dload(*bef), 1, pe_anz, &quell, &ziel);
      if (err || fperror) return;
      acttr++; maxtr--;
    }
  }
  s_spez_vars[MAXTRANS_VAR].datentyp = KEINER | INT_ERLAUBT;    /* MaxTrans wird uninitialisiert */
  set_i_item(&s_spez_vars[ACTTRANS_VAR], acttr, 0);
}

/**************************************************************************
 ***                  Funktion do_store
 ***
 *** fuehrt Befehl 'STORE vc1 TO verg' fuer pe_nr == 0 bzw.
 ***               'STORE vc1 TO verg PE pe_nr' aus
 ***
 **************************************************************************/

int do_store(bef)
STAT *bef;
{ register ITEM *quell, *ziel, *maxit;
  register VARBLOCK *bz;
  register TYP t = ARG_valtyp(STAT_vc1(*bef));
  register long maxtr = (ITEM_typ(s_spez_vars[MAXTRANS_VAR]) == INT)
                        ? s_spez_vars[MAXTRANS_VAR].inhalt.i_val : (long)pe_anz;
  register long acttr = 0;
  register ITPTRS *quellits;
  register char *akt_pes;
  register int last_akt, anz_akt_save;

  if (reducing)
  { runerr(texte[114]); return; }
  ziel = (zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp))->itps[0];
  if (err) return;
  quellits = item_ptr(&STAT_vc1(*bef), &dummybs, disp);
  if (err) return;
  bz = zielbs->blocks[0];
  maxit = bz->vars - bz->von + bz->bis;
  anz_akt_save = anz_akt_pes;
  if (last_akt = STAT_pe_nr(*bef))         /* nur ein PE aktivieren */
  { akt_pes = dummy_aktive_pes;
    if ((akt_pes[last_akt - 1] = aktive_pes[last_akt - 1]) == '0')
    { runwarn(texte[231]); anz_akt_pes = 0; }
    else
    { anz_akt_pes = 1; }
  }
  else
  { akt_pes = aktive_pes; last_akt = last_akt_pe;
    vec_bef = TRUE;
  }
  protokoll('L', akt_pes);
  for (pec = 0; pec < last_akt && maxtr; pec++)
  { quell = (* quellits->n_it_func)(quellits);
    if (akt_pes[pec] == '1')
    { if (ziel > maxit)
      { runerr(texte[112]); 
        dummy_aktive_pes[last_akt - 1] = '0';
        anz_akt_pes = anz_akt_save;
        return;
      }
      if (!(ziel->datentyp & ERLAUBT(t)))
      { runerr(texte[84]);
        dummy_aktive_pes[last_akt - 1] = '0';
        anz_akt_pes = anz_akt_save;
        return;
      }
      set_item(ziel, quell, t);
      if (err || fperror)
      { dummy_aktive_pes[last_akt - 1] = '0';
        anz_akt_pes = anz_akt_save;
        return;
      }
      acttr++; maxtr--; ziel++;
    }
  }
  anz_akt_pes = anz_akt_save;
  dummy_aktive_pes[last_akt - 1] = '0';
  s_spez_vars[MAXTRANS_VAR].datentyp = KEINER | INT_ERLAUBT;    /* MaxTrans wird uninitialisiert */
  set_i_item(&s_spez_vars[ACTTRANS_VAR], acttr, 0);
}

/**************************************************************************
 ***                  Funktion do_blockstore
 ***
 *** fuehrt Befehl 'STORE vc1 TO verg AS dload' aus
 ***
 **************************************************************************/

int do_blockstore(bef)
STAT *bef;
{ ITEM *quell, *ziel;
  register ITEM *maxit;
  register VARBLOCK *bq, *bz, **bqpt;
  register long maxtr = (ITEM_typ(s_spez_vars[MAXTRANS_VAR]) == INT)
                        ? s_spez_vars[MAXTRANS_VAR].inhalt.i_val : (long)pe_anz;
  register long acttr = 0;
  register ITPTRS *quellits;
  register int blocklen = STAT_dload(*bef)->typ_zahl[SUMM];

  protokoll('L', aktive_pes);
  if (reducing)
  { runerr(texte[114]); return; }
  ziel = (zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp))->itps[0];
  if (err) return;
  bz = zielbs->blocks[0];
  quellits = item_ptr(&STAT_vc1(*bef), &dummybs, disp);
  if (err) return;
  bqpt = dummybs->blocks;
  maxit = bz->vars - bz->von + bz->bis;
  vec_bef = TRUE;
  for (pec = 0; pec < last_akt_pe && maxtr; pec++, bqpt += dummybs->inc)
  { quell = (* quellits->n_it_func)(quellits);
    if (aktive_pes[pec] == '1')
    { bq = *bqpt;
      if (ziel > maxit - blocklen + 1)
      { runerr(texte[112]); return; }
      if ((quell - bq->vars) / pe_anz + bq->von + blocklen - 1 > bq->bis)
      { runerr(texte[236]); return; }
      move_as_decl(STAT_dload(*bef), pe_anz, 1, &quell, &ziel);
      if (err || fperror) return;
      acttr++; maxtr--;
    }
  }
  s_spez_vars[MAXTRANS_VAR].datentyp = KEINER | INT_ERLAUBT;    /* MaxTrans wird uninitialisiert */
  set_i_item(&s_spez_vars[ACTTRANS_VAR], acttr, 0);
}

