/*****************************************************************************
  Project: PARZ - Parallel Intermediate Code Debugger/Interpreter
  ----------------------------------------------------------------------------
  Release      : 1
  Project Part : Simulation
  Filename     : do_4fu.c       
  SCCS-Path    : /tmp_mnt/user/sembach/parz/v2/SCCS/s.do_4fu.c
  Version      : 1.3 
  Last changed : 9/27/93 at 14:04:49        
  Author       : Frank Sembach
  Creation date: Aug. 92
  ----------------------------------------------------------------------------
  Description  : Funktionen zur Simulation der PARZ-Befehle

******************************************************************************
***      (C) COPYRIGHT University of Stuttgart - All Right Reserved        ***
*****************************************************************************/

static char sccs_id[] = "@(#)do_4fu.c	1.3  9/27/93 PARZ - Simulation (Frank Sembach)";


#include "parzdefs.h"
#include "rundefs.h"
#include "komdefs.h"
#include "y_tab.h"
#include "externs.h"
#include "runexts.h"
#include "toup.h"

extern FILE *yyerfp;            /* Ausgabedatei fuer Fehlermeldungen */

extern char *wert_string();     /* aus Datei print_fu.c */

extern AKTBLOCK *temp_a_block;  /* Zeiger auf Aktivierungsblock,
                                   der durch CALL erzeugt wurde und
                                   von PROC vervollstaendigt werden muss */

extern char *liesstring();
extern unsigned char *hole_string();

/**************************************************************************
 ***            Funktion do_reduce_first
 ***
 *** fuehrt Befehl 'verg := REDUCE FIRST OF vc1' aus
 *** keine PEs aktiv : wenn vc1 == ID : Ergebnis pe_anz + 1
 ***                   sonst Fehlermeldung
 ***
 **************************************************************************/

int do_reduce_first(bef)
STAT *bef;
{ register ITEM *ziel;
  register ITPTRS *quellits;

  protokoll('U', aktive_pes);
  if (reducing)
  { runerr(texte[114]); return; }
  quellits = item_ptr(&STAT_vc1(*bef), &dummybs, disp);
  if (err) return;
  ziel = (zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp))->itps[0];
  if (err) return;
  vec_bef = TRUE;
  for (pec = 0; pec < last_akt_pe && aktive_pes[pec] != '1'; pec++)
  { (* quellits->n_it_func)(quellits); }
  if (pec == last_akt_pe)
  { if (quellits->itps[0] == v_spez_vars + ID_VAR * pe_anz)
    { set_i_item(ziel, (long)pe_anz + 1l, 0); }
    else
    { vec_bef = FALSE; runerr(texte[113]); }
  }
  else
  { set_item(ziel, (* quellits->n_it_func)(quellits), ARG_valtyp(STAT_verg(*bef))); }
}

/**************************************************************************
 ***            Funktion do_reduce_last
 ***
 *** fuehrt Befehl 'verg := REDUCE LAST OF vc1' aus
 *** keine PEs aktiv : wenn vc1 == ID : Ergebnis 0
 ***                   sonst Fehlermeldung
 ***
 **************************************************************************/

int do_reduce_last(bef)
STAT *bef;
{ register ITEM *quell, *ziel;
  register ITPTRS *quellits;

  protokoll('U', aktive_pes);
  if (reducing)
  { runerr(texte[114]); return; }
  quellits = item_ptr(&STAT_vc1(*bef), &dummybs, disp);
  if (err) return;
  ziel = (zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp))->itps[0];
  if (err) return;
  vec_bef = TRUE;
  for (pec = 0, quell = NULL; pec < last_akt_pe; pec++)
  { quell = (* quellits->n_it_func)(quellits); }
  if (!quell)
  { if (quellits->itps[0] == v_spez_vars + ID_VAR * pe_anz)
    { set_i_item(ziel, 0l, 0); }
    else
    { vec_bef = FALSE; runerr(texte[113]); }
  }
  else
  { set_item(ziel, quell, ARG_valtyp(STAT_verg(*bef))); }
}

/**************************************************************************
 ***          Funktionen do_x_reduce_sum
 ***
 *** fuehren Befehl 'verg := REDUCE SUM OF vc1' aus:
 *** x bezeichnet Typ von verg und vc1:
 ***     i : INTEGER
 ***     r : REAL
 *** keine PEs aktiv : Ergebnis 0
 ***
 **************************************************************************/

int do_i_reduce_sum(bef)
STAT *bef;
{ register ITEM *quell, *ziel;
  register ITPTRS *quellits;
  register long erg = 0l;
  register int uflag = 0;

  protokoll('U', aktive_pes);
  if (reducing)
  { runerr(texte[114]); return; }
  quellits = item_ptr(&STAT_vc1(*bef), &dummybs, disp);
  if (err) return;
  ziel = (zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp))->itps[0];
  if (err) return;
  vec_bef = TRUE;
  for (pec = 0; pec < last_akt_pe; pec++)
  { quell = (* quellits->n_it_func)(quellits);
    if (aktive_pes[pec] == '1')
    { uflag |= ini_typ_test(quell->datentyp, INT, 91, 283);
      erg += quell->inhalt.i_val;
    }
  }
  set_i_item(ziel, erg, uflag);
}

int do_r_reduce_sum(bef)
STAT *bef;
{ register ITEM *quell, *ziel;
  register ITPTRS *quellits;
  register int plen = (anz_akt_pes + 1) / 2;
  register float *puffer;
  float erg = 0.0;
  register int uflag = 0;

  protokoll('U', aktive_pes);
  if (reducing)
  { runerr(texte[114]); return; }
  quellits = item_ptr(&STAT_vc1(*bef), &dummybs, disp);
  if (err) return;
  ziel = (zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp))->itps[0];
  if (err) return;
  vec_bef = TRUE;
  
  if (anz_akt_pes)
  { if (!(puffer = (float *)calloc((size_t)plen,(size_t)sizeof(float))))
    { for (pec = 0; pec < last_akt_pe; pec++)
      { quell = (* quellits->n_it_func)(quellits);
        if (aktive_pes[pec] == '1')
        { uflag |= ini_typ_test(quell->datentyp, REAL, 91, 283);
          erg += quell->inhalt.r_val;
        }
      }
      vec_bef = FALSE;
    }
    else
    { register float *pptr, *ples, *panf, *pend;
      register int add_vorw_flag;
  
      for (pec = 0, pptr = puffer, add_vorw_flag = FALSE;
           pec < last_akt_pe;
           pec++)
      { quell = (* quellits->n_it_func)(quellits);
        if (aktive_pes[pec] == '1')
        { uflag |= ini_typ_test(quell->datentyp, REAL, 91, 283);
          if (add_vorw_flag)
          { *(pptr++) += quell->inhalt.r_val; }
          else
          { *pptr = quell->inhalt.r_val; }
          add_vorw_flag = !add_vorw_flag;
        }
      }
      vec_bef = FALSE;
      panf = puffer; pptr = pend = puffer + plen - 1;
      add_vorw_flag = FALSE;
      while(panf < pend)
      { if (add_vorw_flag)
        { for (ples = pptr; ples < pend; *pptr++ = *ples + ples[1], ples += 2);
          if (ples == pend) *pptr = *ples;
          else pptr--;
          pend = pptr;
        }
        else
        { for (ples = pptr; ples > panf; *pptr-- = *ples + ples[-1], ples -= 2);
          if (ples == panf) *pptr = *ples;
          else pptr++;
          panf = pptr;
        }
        add_vorw_flag = !add_vorw_flag;
      }
      erg = *panf;
      free(puffer);
    }
  }
  set_r_item(ziel, erg, uflag);
}

/**************************************************************************
 ***          Funktionen do_x_reduce_prod
 ***
 *** fuehren Befehl 'verg := REDUCE PRODUCT OF vc1' aus:
 *** x bezeichnet Typ von verg und vc1:
 ***     i : INTEGER
 ***     r : REAL
 *** keine PEs aktiv : Ergebnis 1
 ***
 **************************************************************************/

int do_i_reduce_prod(bef)
STAT *bef;
{ register ITEM *quell, *ziel;
  register ITPTRS *quellits;
  register long erg = 1l;
  register int uflag = 0;

  protokoll('U', aktive_pes);
  if (reducing)
  { runerr(texte[114]); return; }
  quellits = item_ptr(&STAT_vc1(*bef), &dummybs, disp);
  if (err) return;
  ziel = (zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp))->itps[0];
  if (err) return;
  vec_bef = TRUE;
  for (pec = 0; pec < last_akt_pe; pec++)
  { quell = (* quellits->n_it_func)(quellits);
    if (aktive_pes[pec] == '1')
    { uflag |= ini_typ_test(quell->datentyp, INT, 91, 283);
      erg *= quell->inhalt.i_val;
    }
  }
  set_i_item(ziel, erg, uflag);
}

int do_r_reduce_prod(bef)
STAT *bef;
{ register ITEM *quell, *ziel;
  register ITPTRS *quellits;
  register int plen = (anz_akt_pes + 1) / 2;
  register float *puffer;
  float erg = 1.0;
  register int uflag = 0;

  protokoll('U', aktive_pes);
  if (reducing)
  { runerr(texte[114]); return; }
  quellits = item_ptr(&STAT_vc1(*bef), &dummybs, disp);
  if (err) return;
  ziel = (zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp))->itps[0];
  if (err) return;
  vec_bef = TRUE;

  if (anz_akt_pes)
  {  if (!(puffer = (float *)calloc((size_t)plen,(size_t)sizeof(float))))
    { for (pec = 0; pec < last_akt_pe; pec++)
      { quell = (* quellits->n_it_func)(quellits);
        if (aktive_pes[pec] == '1')
        { uflag |= ini_typ_test(quell->datentyp, REAL, 91, 283);
          erg *= quell->inhalt.r_val;
        }
      }
      vec_bef = FALSE;
    }
    else
    { register float *pptr, *ples, *panf, *pend;
      register int mul_vorw_flag;
  
      for (pec = 0, pptr = puffer, mul_vorw_flag = FALSE;
           pec < last_akt_pe;
           pec++)
      { quell = (* quellits->n_it_func)(quellits);
        if (aktive_pes[pec] == '1')
        { uflag |= ini_typ_test(quell->datentyp, REAL, 91, 283);
          if (mul_vorw_flag)
          { *(pptr++) *= quell->inhalt.r_val; }
          else
          { *pptr = quell->inhalt.r_val; }
          mul_vorw_flag = !mul_vorw_flag;
        }
      }
      vec_bef = FALSE;
      panf = puffer; pptr = pend = puffer + plen - 1;
      mul_vorw_flag = FALSE;
      while(panf < pend)
      { if (mul_vorw_flag)
        { for (ples = pptr; ples < pend; *pptr++ = *ples * ples[1], ples += 2);
          if (ples == pend) *pptr = *ples;
          else pptr--;
          pend = pptr;
        }
        else
        { for (ples = pptr; ples > panf; *pptr-- = *ples * ples[-1], ples -= 2);
          if (ples == panf) *pptr = *ples;
          else pptr++;
          panf = pptr;
        }
        mul_vorw_flag = !mul_vorw_flag;
      }
      erg = *panf;
      free(puffer);
    }
  }
  set_r_item(ziel, erg, uflag);
}

/**************************************************************************
 ***          Funktionen do_x_reduce_max
 ***
 *** fuehren Befehl 'verg := REDUCE MAX OF vc1' aus:
 *** x bezeichnet Typ von verg und vc1:
 ***     b : BOOLEAN
 ***     c : CHAR
 ***     i : INTEGER
 ***     r : REAL
 *** keine PEs aktiv : Fehlermeldung
 ***
 **************************************************************************/

int do_b_reduce_max(bef)
STAT *bef;
{ register ITEM *quell, *ziel;
  register ITPTRS *quellits;
  register int erg = FALSE;
  register int uflag = 0;

  protokoll('U', aktive_pes);
  if (reducing)
  { runerr(texte[114]); return; }
  quellits = item_ptr(&STAT_vc1(*bef), &dummybs, disp);
  if (err) return;
  ziel = (zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp))->itps[0];
  if (err) return;
  vec_bef = TRUE;
  for (pec = 0; pec < last_akt_pe; pec++)
  { quell = (* quellits->n_it_func)(quellits);
    if (aktive_pes[pec] == '1')
    { uflag |= ini_typ_test(quell->datentyp, BOOL, 91, 283);
      if (quell->inhalt.b_val == TRUE)  /* fruehzeitiger Abbruch moeglich */
      { erg = TRUE; break; }
    }
  }
  if (anz_akt_pes)
  { set_b_item(ziel, erg, uflag); }
  else
  { vec_bef = FALSE; runerr(texte[113]); }
}

int do_c_reduce_max(bef)
STAT *bef;
{ register ITEM *quell, *ziel;
  register ITPTRS *quellits;
  register unsigned char erg;
  register int found = FALSE;
  register int uflag = 0;

  protokoll('U', aktive_pes);
  if (reducing)
  { runerr(texte[114]); return; }
  quellits = item_ptr(&STAT_vc1(*bef), &dummybs, disp);
  if (err) return;
  ziel = (zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp))->itps[0];
  if (err) return;
  vec_bef = TRUE;
  for (pec = 0; pec < last_akt_pe; pec++)
  { quell = (* quellits->n_it_func)(quellits);
    if (aktive_pes[pec] == '1')
    { uflag |= ini_typ_test(quell->datentyp, CHA, 91, 283);
      if (!found || quell->inhalt.c_val > erg)
      { erg = quell->inhalt.c_val; found = TRUE; }
    }
  }
  if (found)
  { set_c_item(ziel, erg, uflag); }
  else
  { vec_bef = FALSE; runerr(texte[113]); }
}

int do_i_reduce_max(bef)
STAT *bef;
{ register ITEM *quell, *ziel;
  register ITPTRS *quellits;
  register long erg;
  register int found = FALSE;
  register int uflag = 0;

  protokoll('U', aktive_pes);
  if (reducing)
  { runerr(texte[114]); return; }
  quellits = item_ptr(&STAT_vc1(*bef), &dummybs, disp);
  if (err) return;
  ziel = (zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp))->itps[0];
  if (err) return;
  vec_bef = TRUE;
  for (pec = 0; pec < last_akt_pe; pec++)
  { quell = (* quellits->n_it_func)(quellits);
    if (aktive_pes[pec] == '1')
    { uflag |= ini_typ_test(quell->datentyp, INT, 91, 283);
      if (!found || quell->inhalt.i_val > erg)
      { erg = quell->inhalt.i_val; found = TRUE; }
    }
  }
  if (found)
  { set_i_item(ziel, erg, uflag); }
  else
  { vec_bef = FALSE; runerr(texte[113]); }
}

int do_r_reduce_max(bef)
STAT *bef;
{ register ITEM *quell, *ziel;
  register ITPTRS *quellits;
  float erg;
  register int found = FALSE;
  register int uflag = 0;

  protokoll('U', aktive_pes);
  if (reducing)
  { runerr(texte[114]); return; }
  quellits = item_ptr(&STAT_vc1(*bef), &dummybs, disp);
  if (err) return;
  ziel = (zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp))->itps[0];
  if (err) return;
  vec_bef = TRUE;
  for (pec = 0; pec < last_akt_pe; pec++)
  { quell = (* quellits->n_it_func)(quellits);
    if (aktive_pes[pec] == '1')
    { uflag |= ini_typ_test(quell->datentyp, REAL, 91, 283);
      if (!found || quell->inhalt.r_val > erg)
      { erg = quell->inhalt.r_val; found = TRUE; }
    }
  }
  if (found)
  { set_r_item(ziel, erg, uflag); }
  else
  { vec_bef = FALSE; runerr(texte[113]); }
}

/**************************************************************************
 ***          Funktionen do_x_reduce_min
 ***
 *** fuehren Befehl 'verg := REDUCE MIN OF vc1' aus:
 *** x bezeichnet Typ von verg und vc1:
 ***     b : BOOLEAN
 ***     c : CHAR
 ***     i : INTEGER
 ***     r : REAL
 *** keine PEs aktiv : Fehlermeldung
 ***
 **************************************************************************/

int do_b_reduce_min(bef)
STAT *bef;
{ register ITEM *quell, *ziel;
  register ITPTRS *quellits;
  register int erg = TRUE;
  register int uflag = 0;

  protokoll('U', aktive_pes);
  if (reducing)
  { runerr(texte[114]); return; }
  quellits = item_ptr(&STAT_vc1(*bef), &dummybs, disp);
  if (err) return;
  ziel = (zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp))->itps[0];
  if (err) return;
  vec_bef = TRUE;
  for (pec = 0; pec < last_akt_pe; pec++)
  { quell = (* quellits->n_it_func)(quellits);
    if (aktive_pes[pec] == '1')
    { uflag |= ini_typ_test(quell->datentyp, BOOL, 91, 283);
      if (quell->inhalt.b_val == FALSE)    /* fruehzeitiger Abbruch moeglich */
      { erg = FALSE; break; }
    }
  }
  if (anz_akt_pes)
  { set_b_item(ziel, erg, uflag); }
  else
  { vec_bef = FALSE; runerr(texte[113]); }
}

int do_c_reduce_min(bef)
STAT *bef;
{ register ITEM *quell, *ziel;
  register ITPTRS *quellits;
  register unsigned char erg;
  register int found = FALSE;
  register int uflag = 0;

  protokoll('U', aktive_pes);
  if (reducing)
  { runerr(texte[114]); return; }
  quellits = item_ptr(&STAT_vc1(*bef), &dummybs, disp);
  if (err) return;
  ziel = (zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp))->itps[0];
  if (err) return;
  vec_bef = TRUE;
  for (pec = 0; pec < last_akt_pe; pec++)
  { quell = (* quellits->n_it_func)(quellits);
    if (aktive_pes[pec] == '1')
    { uflag |= ini_typ_test(quell->datentyp, CHA, 91, 283);
      if (!found || quell->inhalt.c_val < erg)
      { erg = quell->inhalt.c_val; found = TRUE; }
    }
  }
  if (found)
  { set_c_item(ziel, erg, uflag); }
  else
  { vec_bef = FALSE; runerr(texte[113]); }
}

int do_i_reduce_min(bef)
STAT *bef;
{ register ITEM *quell, *ziel;
  register ITPTRS *quellits;
  register long erg;
  register int found = FALSE;
  register int uflag = 0;

  protokoll('U', aktive_pes);
  if (reducing)
  { runerr(texte[114]); return; }
  quellits = item_ptr(&STAT_vc1(*bef), &dummybs, disp);
  if (err) return;
  ziel = (zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp))->itps[0];
  if (err) return;
  vec_bef = TRUE;
  for (pec = 0; pec < last_akt_pe; pec++)
  { quell = (* quellits->n_it_func)(quellits);
    if (aktive_pes[pec] == '1')
    { uflag |= ini_typ_test(quell->datentyp, INT, 91, 283);
      if (!found || quell->inhalt.i_val < erg)
      { erg = quell->inhalt.i_val; found = TRUE; }
    }
  }
  if (found)
  { set_i_item(ziel, erg, uflag); }
  else
  { vec_bef = FALSE; runerr(texte[113]); }
}

int do_r_reduce_min(bef)
STAT *bef;
{ register ITEM *quell, *ziel;
  register ITPTRS *quellits;
  float erg;
  register int found = FALSE;
  register int uflag = 0;

  protokoll('U', aktive_pes);
  if (reducing)
  { runerr(texte[114]); return; }
  quellits = item_ptr(&STAT_vc1(*bef), &dummybs, disp);
  if (err) return;
  ziel = (zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp))->itps[0];
  if (err) return;
  vec_bef = TRUE;
  for (pec = 0; pec < last_akt_pe; pec++)
  { quell = (* quellits->n_it_func)(quellits);
    if (aktive_pes[pec] == '1')
    { uflag |= ini_typ_test(quell->datentyp, REAL, 91, 283);
      if (!found || quell->inhalt.r_val < erg)
      { erg = quell->inhalt.r_val; found = TRUE; }
    }
  }
  if (found)
  { set_r_item(ziel, erg, uflag); }
  else
  { vec_bef = FALSE; runerr(texte[113]); }
}

/**************************************************************************
 ***          Funktion do_b_reduce_and
 ***
 *** fuehrt Befehl 'verg := REDUCE AND OF vc1' aus:
 *** keine PEs aktiv : Ergebnis = TRUE
 ***
 **************************************************************************/

int do_b_reduce_and(bef)
STAT *bef;
{ register ITEM *quell, *ziel;
  register ITPTRS *quellits;
  register int erg;
  register int uflag = 0;

  protokoll('U', aktive_pes);
  if (reducing)
  { runerr(texte[114]); return; }
  quellits = item_ptr(&STAT_vc1(*bef), &dummybs, disp);
  if (err) return;
  ziel = (zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp))->itps[0];
  if (err) return;
  vec_bef = TRUE;
  erg = TRUE;
  for (pec = 0; pec < last_akt_pe; pec++)
  { quell = (* quellits->n_it_func)(quellits);
    if (aktive_pes[pec] == '1')
    { uflag |= ini_typ_test(quell->datentyp, BOOL, 91, 283);
      if (quell->inhalt.b_val == FALSE)    /* fruehzeitiger Abbruch moeglich */
      { erg = FALSE; break; }
    }
  }
  set_b_item(ziel, erg, uflag);
}

/**************************************************************************
 ***          Funktion do_b_reduce_or
 ***
 *** fuehrt Befehl 'verg := REDUCE OR OF vc1' aus:
 *** keine PEs aktiv : Ergebnis = FALSE
 ***
 **************************************************************************/

int do_b_reduce_or(bef)
STAT *bef;
{ register ITEM *quell, *ziel;
  register ITPTRS *quellits;
  register int erg;
  register int uflag = 0;

  protokoll('U', aktive_pes);
  if (reducing)
  { runerr(texte[114]); return; }
  quellits = item_ptr(&STAT_vc1(*bef), &dummybs, disp);
  if (err) return;
  ziel = (zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp))->itps[0];
  if (err) return;
  vec_bef = TRUE;
  erg = FALSE;
  for (pec = 0; pec < last_akt_pe; pec++)
  { quell = (* quellits->n_it_func)(quellits);
    if (aktive_pes[pec] == '1')
    { uflag |= ini_typ_test(quell->datentyp, BOOL, 91, 283);
      if (quell->inhalt.b_val == TRUE)  /* fruehzeitiger Abbruch moeglich */
      { erg = TRUE; break; }
    }
  }
  set_b_item(ziel, erg, uflag);
}

/**************************************************************************
 ***           Funktion red_proc_call
 ***
 *** Sichert den Zustand fuer REDUCE und ruft die Operatorfunktion auf.
 *** Zwei Werte muessen schon auf dem skalarenb Stack liegen.
 *** bef : REDUCE-Befehl, der Operatoradresse enthaelt
 ***
 **************************************************************************/

red_proc_call(bef)
STAT *bef;
{ register int call_count_save, pc_save;

  reducing = TRUE;
  temp_a_block = new_aktblock(pc + 1, NULL, TRUE);
  called = TRUE;
  call_count_save = call_count;
  pc_save = pc;
  exit_code = start(STAT_red_fct(*bef));
  reducing = FALSE;
  call_count = call_count_save;
  pc = pc_save;
}

/**************************************************************************
 ***           Funktion do_proc_reduce
 ***
 *** fuehrt Befehl 'verg := REDUCE red_fct OF vc1' aus
 *** keine PEs aktiv : Fehlermeldung
 ***
 **************************************************************************/

int do_proc_reduce(bef)
STAT *bef;
{ register ITEM *quell, *ziel;
  register ITPTRS *quellits;
  register TYP t = ARG_valtyp(STAT_verg(*bef));
  register int uflag = 0;

  if (reducing)
  { runerr(texte[114]); return; }
  protokoll('U', aktive_pes);
  quellits = item_ptr(&STAT_vc1(*bef), &dummybs, disp);
  if (err) return;
  ziel = (zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp))->itps[0];
  if (err) return;
  vec_bef = TRUE;

  if (anz_akt_pes)
  { register int push_vorw_flag, reserve;
    register ITEM *last_item;
    ITEM popit;
    register int anz_akt_save = anz_akt_pes;
    register int last_akt_save = last_akt_pe;
    register char *akt_pes_save = aktive_pes;

    if (!(aktive_pes = calloc((size_t)(pe_anz + 1), (size_t)sizeof(char))))
    { runerr(texte[232]); return; }
    strcpy(aktive_pes, akt_pes_save);
    for (pec = 0, push_vorw_flag = FALSE; pec < last_akt_pe; pec++)
    { quell = (* quellits->n_it_func)(quellits);
      if (aktive_pes[pec] == '1')
      { uflag |= ini_typ_test(quell->datentyp, t, 91, 283);
        if (push_vorw_flag)
        { STACK_push(last_item, vec_stacks + pec, t);
          if (err || fperror) return;
          STACK_push(quell, vec_stacks + pec, t);
          if (err || fperror) return;
        }
        else
        { aktive_pes[pec] = '0';
          last_item = quell;
        }
        push_vorw_flag = !push_vorw_flag;
      }
    }
    if (push_vorw_flag)
    { STACK_push(last_item, vec_stacks + --pec, t);
      if (err || fperror) return;
      reserve = pec;
    }
    else
    { reserve = 0; }
    push_vorw_flag = vec_bef = FALSE;
    do
    { register int count, inc, penr, palt;

      last_akt_pe = last(aktive_pes, &anz_akt_pes);
      if (anz_akt_pes == 0) break;
      red_proc_call(bef);
      switch (exit_code)
      { case DO_STOP:
        case DO_LOAD:
        case DO_COMPILE: return;
      }
      if (reserve)
      { aktive_pes[reserve] = '1';
        last_akt_pe = last(aktive_pes, &anz_akt_pes);
      }
      if (err || fperror) return;
      for (count = last_akt_pe, penr = push_vorw_flag ? 0 : last_akt_pe - 1,
           popit.datentyp = 0,  inc = push_vorw_flag ? 1 : -1;
           count;
           count--, penr += inc)
      { if (aktive_pes[penr] == '1')
        { if (popit.datentyp)
          { STACK_push(&popit, vec_stacks + penr, t);
            if (err || fperror) return;
            popit.datentyp = 0;
          }
          else
          { aktive_pes[palt = penr] = '0';
            popit.datentyp = KEINER | ERLAUBT(t);
            STACK_pop(&popit, vec_stacks + penr, t);
            if (err || fperror) return;
          }
        }
      }
      if (popit.datentyp)
      { reserve = palt;
        STACK_push(&popit, vec_stacks + palt, t);
        if (err || fperror) return;
      }
      else
      { reserve = 0; }
      push_vorw_flag = !push_vorw_flag;
    } while (1);
    STACK_pop(&popit, vec_stacks + reserve, t);
    popit.datentyp |= (uflag & UNDEF_ERG);
    set_item(ziel, &popit, t);
    anz_akt_pes = anz_akt_save;
    last_akt_pe = last_akt_save;
    free(aktive_pes);
    aktive_pes = akt_pes_save;
  }
  else
  { vec_bef = FALSE; runerr(texte[113]); }
}

/**************************************************************************
 ***           Funktion STACK_push
 ***
 *** schiebt '*it' auf den Stack '*stackptr'.
 *** Das gepushte ITEM erhaelt den aktuellen Typ 't'.
 *** Legt wenn noetig neuen Stackblock an.
 ***
 **************************************************************************/

STACK_push(it, stackptr, t)
ITEM *it;
STACK **stackptr;
TYP t;
{ register int st;

  if (! *stackptr)              /* Stack existiert nicht : neuen erzeugen */
  { if (!(*stackptr = (STACK *)calloc((size_t)1,(size_t)sizeof(STACK))))
    { runerr(texte[61]); return; }
  }
  st = (*stackptr)->top++;      /* top zeigt auf naechste freie Speicherstelle */
  if (st >= STACKPARTLEN)       /* Stack ist voll : neuen Block anhaengen */
  { register STACK *neustack;

    if (!(neustack = (STACK *)calloc((size_t)1,(size_t)sizeof(STACK))))
    { runerr(texte[61]); return; }
    neustack->n_stack_part = *stackptr;
    *stackptr = neustack;
    st = (*stackptr)->top++;
  }
  (*stackptr)->speicher[st].datentyp = t | ERLAUBT(t);
  set_item(&(*stackptr)->speicher[st], it, t);
}

/**************************************************************************
 ***                Funktion STACK_pop
 ***
 *** holt '*it' vom Stack '*stackptr'
 *** Der Typ auf dem Stack muss 't' sein, sonst Fehlermeldung.
 *** Verkuerzt wenn noetig Stack um einen Stackblock
 ***
 **************************************************************************/

STACK_pop(it, stackptr, t)
ITEM *it;
STACK **stackptr;
TYP t;
{ register int st;

  if (! *stackptr)              /* Stack existiert nicht : pop unmoeglich */
  { runerr(texte[123]); return; }
  st = --(*stackptr)->top;      /* zu holende Zelle liegt um 1 unter top */
  if (st < 0)
  { register STACK *altstack = (*stackptr)->n_stack_part;

    free(*stackptr);
    *stackptr = altstack;
    if (! *stackptr)            /* Stack ist leer : pop unmoeglich */
    { runerr(texte[123]); return; }
    st = ((*stackptr)->top -= 2);  /* top steht bei vollem Stack
                                      auf STACKPARTLEN + 1 */
  }
  if (ITEM_typ((*stackptr)->speicher[st]) != t)
  { runerr(texte[124]); return; }
  set_item(it, &(*stackptr)->speicher[st], t);
}

/**************************************************************************
 ***       Funktion do_pushs
 ***
 *** fuehrt Befehl 'PUSHS vc1' aus (Push auf skalaren Stack)
 ***
 **************************************************************************/

int do_pushs(bef)
STAT *bef;
{ register ITEM *quell = item_ptr(&STAT_vc1(*bef), &dummybs, disp)->itps[0];
  register TYP t = ARG_valtyp(STAT_vc1(*bef));

  protokoll('s', NULL);
  if (err) return;
  ini_typ_test(quell->datentyp, t, 91, 283);
  STACK_push(quell, &scal_stack, t);
}

/**************************************************************************
 ***                  Funktionen do_t_pushv
 ***
 *** fuehren Befehl 'PUSHV vc1' aus (Push auf Vektorstacks):
 *** t = s : Skalare Daten, t = v : Vektordaten
 ***
 **************************************************************************/

int do_s_pushv(bef)
STAT *bef;
{ register ITEM *quell;
  register STACK **stptr = vec_stacks;
  register TYP t = ARG_valtyp(STAT_vc1(*bef));

  protokoll('S', aktive_pes);
  quell = item_ptr(&STAT_vc1(*bef), &dummybs, disp)->itps[0];
  if (err) return;
  ini_typ_test(quell->datentyp, t, 91, 283);
  vec_bef = TRUE;
  for (pec = 0; pec < last_akt_pe; pec++, stptr++)
  { if (aktive_pes[pec] == '1')
    { STACK_push(quell, stptr, t);
      if (err) return;
    }
  }
}

int do_v_pushv(bef)
STAT *bef;
{ register ITEM *quell;
  register STACK **stptr = vec_stacks;
  register ITPTRS *quellits;
  register TYP t = ARG_valtyp(STAT_vc1(*bef));

  protokoll('S', aktive_pes);
  quellits = item_ptr(&STAT_vc1(*bef), &dummybs, disp);
  if (err) return;
  vec_bef = TRUE;
  for (pec = 0; pec < last_akt_pe; pec++, stptr++)
  { quell = (* quellits->n_it_func)(quellits);
    if (aktive_pes[pec] == '1')
    { ini_typ_test(quell->datentyp, t, 91, 283);
      STACK_push(quell, stptr, t);
      if (err) return;
    }
  }
}

/**************************************************************************
 ***                    Funktionen do_t_pops
 ***
 *** fuehren Befehl 'POPS verg' aus (Pop vom skalaren Stack)
 *** t = s : Skalare Zielvariable, t = v : Vektorielle Zielvariable
 ***
 **************************************************************************/

int do_s_pops(bef)
STAT *bef;
{ register ITEM *ziel = (zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp))->itps[0];

  protokoll('s', NULL);
  if (err) return;
  STACK_pop(ziel, &scal_stack, ARG_valtyp(STAT_verg(*bef)));
}

int do_v_pops(bef)
STAT *bef;
{ ITEM zwisch;
  register ITEM  *ziel;
  register TYP t = ARG_valtyp(STAT_verg(*bef));

  protokoll('S', aktive_pes);
  zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp);
  zwisch.datentyp = t | ERLAUBT(t);
  STACK_pop(&zwisch, &scal_stack, t);
  if (err) return;
  vec_bef = TRUE;
  for (pec = 0; pec < last_akt_pe; pec++)
  { ziel = (* zielits->n_it_func)(zielits);
    if (aktive_pes[pec] == '1')
    { set_item(ziel, &zwisch, t);
      if (err) return;
    }
  }
}

/**************************************************************************
 ***                   Funktion do_popv
 ***
 *** fuehrt Befehl 'POPV verg' aus (Pop von Vektorstacks)
 ***
 **************************************************************************/

int do_popv(bef)
STAT *bef;
{ register ITEM *ziel;
  register STACK **stptr = vec_stacks;
  register TYP t = ARG_valtyp(STAT_verg(*bef));

  protokoll('S', aktive_pes);
  zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp);
  if (err) return;
  vec_bef = TRUE;
  for (pec = 0; pec < last_akt_pe; pec++, stptr++)
  { ziel = (* zielits->n_it_func)(zielits);
    if (aktive_pes[pec] == '1')
    { STACK_pop(ziel, stptr, t);
      if (err) return;
    }
  }
}

/**************************************************************************
 ***         Funktion do_call
 ***
 *** fuehrt Befehl 'CALL spr_ziel' aus
 ***
 **************************************************************************/

int do_call(bef)
STAT *bef;
{ protokoll('c', NULL);
  temp_a_block = new_aktblock(pc + 1, NULL, TRUE);
  called = TRUE;
  call_count++;
  pc = STAT_spr_ziel(*bef) - 1;
}

/**************************************************************************
 ***                  Funktion do_proc
 ***
 *** fuehrt Befehl 'PROC proclev dscal dvec' aus
 ***
 **************************************************************************/

int do_proc(bef)
STAT *bef;
{ protokoll('f', NULL);
  if (!called)
  { runerr(texte[119]); return; }
  called = FALSE;
  fill_aktblock(temp_a_block, STAT_proclev(*bef), pc, STAT_dscal(*bef), STAT_dvec(*bef));
  vec_bef = TRUE;
}

/**************************************************************************
 ***                      Funktion newpar
 ***
 *** setzt 'aktive_pes' auf 'parstr', wo die Menge der maximal aktiven PEs
 *** des aktuellen Aktivierungsblocks dies zulaesst.
 *** Gibt Warnung aus, wenn mehr PEs als moeglich aktiviert werden sollen
 ***
 **************************************************************************/

newpar(parstr)
char *parstr;
{ register char *alt = disp[akt_tiefe]->start_aktiv;
  register char *neu = parstr;
  register char *p = aktive_pes;

  for (; *alt; alt++, neu++, p++)
  { if (*alt == '0' && *neu == '1')
    { *p = '0';
      runwarn(texte[107]);
      break;
    }
    else
    { *p = *neu; }
  }
  for (; *alt; alt++, neu++, p++)       /* Fortsetzung nach 1. Warnung  */
  { *p = (*alt == '0' && *neu == '1') ? '0' : *neu; }
  last_akt_pe = last(aktive_pes,&anz_akt_pes);
}

/**************************************************************************
 ***          Funktion do_parbit
 ***
 *** fuehrt Befehl 'PARALLEL parbits' aus
 ***
 **************************************************************************/

int do_parbit(bef)
STAT *bef;
{ protokoll('M', aktive_pes);
  if (reducing)
  { runerr(texte[114]); return; }
  newpar(STAT_parbits(*bef));
}

/**************************************************************************
 ***                   Funktion do_parvar
 ***
 *** fuehrt Befehl 'PARALLEL vc1' aus
 *** 'vc1' muss Anfang von BOOLEAN-Array der Laenge 'pe_anz' sein,
 *** sonst Fehlermeldung
 ***
 **************************************************************************/

int do_parvar(bef)
STAT *bef;
{ register ITEM *it;
  register int i;
  register char *neu;
  register VARBLOCK *bl;

  protokoll('M', aktive_pes);
  if (reducing)
  { runerr(texte[114]); return; }
  it = item_ptr(&STAT_vc1(*bef), &dummybs, disp)->itps[0];
  if (err) return;
  bl = dummybs->blocks[0];
  if (!(neu = calloc((size_t)(pe_anz + 1), (size_t)sizeof(char))))
  { runerr(texte[4]); return; }
  if ((it - bl->vars) + bl->von + pe_anz - 1 > bl->bis)
  { runerr(texte[126]); free(neu); return; }
  for (i = 0; i < pe_anz; i++, it++)
  { if (!(it->datentyp & BOOL_ERLAUBT))
    { runerr(texte[127]); free(neu); return; }
    ini_typ_test(it->datentyp, BOOL, 91, 283);
    neu[i] = (it->inhalt.b_val) ? '1' : '0';
  }
  newpar(neu); free(neu);
}

/**************************************************************************
 ***                   Funktion do_t_initset
 ***
 *** fuehrt Befehl 'verg := INITSET setbits' aus
 *** 'verg' muss Anfang von BOOLEAN-Array sein, sonst Fehlermeldung
 *** t = s : skalares Zielset, t = v : vektorielles Zielset
 ***
 **************************************************************************/

int do_s_initset(bef)
STAT *bef;
{ register ITEM *ziel;
  register char *bits;

  protokoll('b', NULL);
  ziel = (zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp))->itps[0];
  if (err) return;
  bits = STAT_setbits(*bef);
  { register VARBLOCK *bl = zielbs->blocks[0];

    if ((ziel - bl->vars) + bl->von + strlen(bits) - 1 > bl->bis)
    { runerr(texte[318]); return; }
    for (; *bits; bits++, ziel++)
    { if (!(ziel->datentyp & BOOL_ERLAUBT))
      { runerr(texte[319]); return; }
      set_b_item(ziel, *bits == '1', 0);
    }
  }
}

int do_v_initset(bef)
STAT *bef;
{ register ITEM *ziel;
  register VARBLOCK **blpt;

  protokoll('B', aktive_pes);
  zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp);
  if (err) return;
  blpt = zielbs->blocks;
  vec_bef = TRUE;
  for (pec = 0; pec < last_akt_pe; pec++, blpt += zielbs->inc)
  { ziel = (* zielits->n_it_func)(zielits);
    if (aktive_pes[pec] == '1')
    { register char *bits = STAT_setbits(*bef);
      register VARBLOCK *bl = *blpt;
     
      if ((ziel - bl->vars) / pe_anz + bl->von + strlen(bits) - 1 > bl->bis)
      { runerr(texte[318]); return; }
      for (; *bits; bits++, ziel += pe_anz)
      { if (!(ziel->datentyp & BOOL_ERLAUBT))
        { runerr(texte[319]); return; }
        set_b_item(ziel, *bits == '1', 0);
      }
    }
  }
}

