/*****************************************************************************
  Project: PARZ - Parallel Intermediate Code Debugger/Interpreter
  ----------------------------------------------------------------------------
  Release      : 1
  Project Part : Parser
  Filename     : parse_fu.c       
  SCCS-Path    : /tmp_mnt/user/sembach/parz/v2/SCCS/s.parse_fu.c
  Version      : 1.3 
  Last changed : 9/27/93 at 14:04:57        
  Author       : Frank Sembach
  Creation date: Aug. 92
  ----------------------------------------------------------------------------
  Description  : Parser-Hilfsfunktionen fuer PARZ

******************************************************************************
***      (C) COPYRIGHT University of Stuttgart - All Right Reserved        ***
*****************************************************************************/

static char sccs_id[] = "@(#)parse_fu.c	1.3  9/27/93 PARZ - Parser (Frank Sembach)";


#include "parzdefs.h"
#include "komdefs.h"
#include "y_tab.h"

/*********************************************/
#include "externs.h"
#include "rundefs.h"
#include "runexts.h"
#include "komexts.h"
#include "funcs.h"

/* externs aus parz.l :  */
extern char *lies_comm(), *lies_zend();

extern char ausgeben;       /* Flag : Naechste Fehlermeldung ausgeben */
extern char bef_lesen;      /* Flag : Befehlstoken erwartet (Kommandointerp. ) */

extern FILE *yyin;          /* Eingabefile */
extern FILE *yyerfp;        /* File fuer Fehlermeldungen */
extern int yylineno;        /* Zeilenzaehler */

/*********************************************/

extern int yynerrs;         /* Zaehler fuer Fehlermeldungen */
#ifdef YYDEBUG
extern int yydebug;
#endif

extern char lesend, nospez;        /* Art des naechsten Zugriffs auf eine Variable */

extern char do_step;                   /* parser hat "#\n" zurueckgegeben */
extern char scanner_fehler;    /* scanner hat in Zeile Fehler gemeldet */

extern int flags;                      /* Information ueber gelesene Schluesselworte */
extern int nach_end;                   /* Zaehlt Befehle nach dem letzten END-Befehl */
extern int last_label;                 /* letztes gelesenes Label */
extern int akt_max;                    /* maximale verfuegbare Programmadresse */

#define HEAP_ADRS 1
#define VEC_ADRS 2
extern char adrsort;                   /* Art der Adressen fuer memel */

extern STAT *stat_ptr;                 /* Zeiger auf momentan erzeugten Befehl */

extern struct
{ int (* do_f[2][REAL + 1])();
  int (* print_f)();
} op_funcs[OR - PLUS + 1];      /* Zeiger auf Funktionen fuer unaere und binaere
                                   Operatoren */

extern struct
{ int (* do_rf[2])(), (* print_rf)();
} rel_op_funcs[GE - EQ + 1];           /* Funktionen fuer Vergleichsbefehle */

extern int (* assign_do_funcs[2][REAL + 1][STR + 1])();       /* Funktionen fuer Zuweisung */

extern int (* random_do_funcs[2][STR + 1])();         /* Funktionen fuer RANDOM */

extern int (* strcmp_do_funcs[2][STR + 1][STR + 1])();        /* Funktionen fuer STRCMP */

extern int (* red_do_funcs[STR + 1][MIN - AND + 1])();      /* Funktionen fuer REDUCE */

extern int (* read_do_funcs[2][REAL + 1])();     /* Funktionen fuer READ */

extern int (* write2_do_funcs[2][STR + 1])();    /* Funktionen fuer WRITE (2 Argumente) */

extern int (* draw2_do_funcs[STR + 1])();    /* Funktionen fuer DRAW (2 Argumente) */

extern int (* writeln2_do_funcs[2][STR + 1])();  /* Funktionen fuer WRITELN (2 Argumente) */

extern int red_typs[MIN - AND + 1];           /* erlaubte Typen fuer Reduktionsfunktionen : */

extern char *ssstr[];           /* Texte fuer Singlestep-/Breakpointmodus */
extern char *protstr[];         /* Texte fuer Protokollmodus */

extern char *promptstr[2];

extern int graph_bef_exist;

/**************************************************************************
 ***                       Funktion init_prog
 ***
 *** bereitet den Programmspeicher zur Aufnahme der Programmzeilen vor
 ***
 **************************************************************************/

init_prog()
{ stat_anz = max_lev = nach_end = last_label = 0;
  graph_bef_exist = FALSE;
  if (!(stat_ptr = programm = (STAT *)calloc((size_t)STARTPROGLEN,(size_t)sizeof(STAT))))
  { fatal(texte[60]); return; }
  akt_max = STARTPROGLEN - 1;
}

/**************************************************************************
 ***                       Funktion create_pes
 ***
 *** initialisiert Stacks und Portbeschreibungen fuer die Prozessoreinheiten:
 *** richtet fuer jeden Stack NULL-Zeiger ein
 *** setzt fuer jeden Port 'ziel' = NULL
 ***
 **************************************************************************/

create_pes()
{ if (pe_anz > 0)
  { if (!(vec_stacks = (STACK **)calloc((size_t)pe_anz, (size_t)sizeof(STACK *))))
    { fatal(texte[61]); return; }
    if (port_anz > 0)
    { if (!(portarray = (PORT *)calloc((size_t)(pe_anz * port_anz),
                                       (size_t)sizeof(PORT))))
      { fatal(texte[62]); }
    }
  }
}

/**************************************************************************
 ***                       Funktion loesch_prog
 ***
 *** gibt Speicherplatz des Programmtexts in '*prog' zurueck
 *** *anz : Zahl der Befehle in '*prog'
 *** free_flag : gibt an, ob Programmspeicher freigegeben wird
 *** Bei Rueckkehr gilt '*anz == 0' und '*prog == NULL'
 ***
 **************************************************************************/

loesch_prog(prog, anz, free_flag)
STAT **prog;
int *anz, free_flag;
{ register STAT *ptr = *prog;
  register int z;

  for (z = 0; z < *anz; z++, ptr++)
  { int (* df)() = STAT_print_func(*ptr);
    int (* do_f)() = STAT_do_func(*ptr);
                /* Deklarationen freigeben */
    if (df == print_new)
    { loesch_decl(STAT_dnew(*ptr));
    }
    else if (df == print_blockmove ||
             df == print_blockequal ||
             df == print_debug ||
             df == print_trace)
    { loesch_decl(STAT_dblock(*ptr));
    }
    else if (df == print_proc)
    { loesch_decl(STAT_dscal(*ptr));
      loesch_decl(STAT_dvec(*ptr));
    }
    else if (df == print_blockload ||
             df == print_blockstore)
    { loesch_decl(STAT_dload(*ptr));
    }
                /* Bitstrings freigeben */
    if (df == print_parbit && STAT_parbits(*ptr))
    { free(STAT_parbits(*ptr));
    }
    if (df == print_initset)
    { free(STAT_setbits(*ptr));
    }
                /* strings und SIZE-Deklarationen in Argumenten freigeben */
    if (do_f == do_s_ifcall ||
        do_f == do_v_ifcall ||
        do_f == do_ifgoto ||
        do_f == do_s_whilecall ||
        do_f == do_v_whilecall ||
        df == print_1_write ||
        df == print_1_draw ||
        df == print_1_writeln ||
        df == print_pushs ||
        df == print_pushv ||
        df == print_parvar ||
        df == print_openinput ||
        df == print_openoutput ||
        df == print_errorcall ||
        df == print_setcolor ||
        df == print_selectw ||
        df == print_closew ||
        df == print_notrace)
    { loesch_arg(&STAT_vc1(*ptr));
    }
    else if (do_f == do_s2_ifcall ||
             do_f == do_v2_ifcall ||
             do_f == do_2_ifgoto ||
             do_f == do_s2_whilecall ||
             do_f == do_v2_whilecall ||
             df == print_2_write ||
             df == print_2_draw ||
             df == print_2_writeln ||
             df == print_moveto ||
             df == print_lineto ||
             df == print_setpixel)
    { loesch_arg(&STAT_vc1(*ptr));
      loesch_arg(&STAT_vc2(*ptr));
    }
    else if (df == print_3_write ||
             df == print_3_draw ||
             df == print_3_writeln ||
             df == print_wsize)
    { loesch_arg(&STAT_vc1(*ptr));
      loesch_arg(&STAT_vc2(*ptr));
      loesch_arg(&STAT_vc3(*ptr));
    }
    else if (df == print_propagate ||
             df == print_connect ||
             df == print_biconnect)
    { loesch_arg(&STAT_vc1(*ptr));
      loesch_arg(&STAT_out_port(*ptr));
      loesch_arg(&STAT_in_port(*ptr));
    }
    else if (df == print_inconnected)
    { loesch_arg(&STAT_verg(*ptr));
      loesch_arg(&STAT_cin_port(*ptr));
    }
    else if (df == print_outconnected)
    { loesch_arg(&STAT_verg(*ptr));
      loesch_arg(&STAT_out_port(*ptr));
    }
    else if (df == print_line_inconnected ||
             df == print_line_outconnected)
    { loesch_arg(&STAT_verg(*ptr));
      loesch_arg(&STAT_out_port(*ptr));
      loesch_arg(&STAT_cin_port(*ptr));
    }
    else if (df == print_zuw ||
             df == print_minus ||
             df == print_not ||
             df == print_sqrt ||
             df == print_exp ||
             df == print_ln ||
             df == print_sin ||
             df == print_cos ||
             df == print_tan ||
             df == print_arcsin ||
             df == print_arccos ||
             df == print_arctan ||
             df == print_abs ||
             df == print_read_string ||
             df == print_load ||
             df == print_blockload ||
             df == print_store ||
             df == print_blockstore ||
             df == print_op_reduce ||
             df == print_proc_reduce ||
             df == print_to_propagate ||
             df == print_send ||
             df == print_receive ||
             df == print_redproc_propagate ||
             df == print_redproc_send ||
             df == print_redproc_receive)
    { loesch_arg(&STAT_vc1(*ptr));
      loesch_arg(&STAT_verg(*ptr));
    }
    else if (df == print_new ||
             df == print_status ||
             df == print_random ||
             df == print_read ||
             df == print_pops ||
             df == print_popv ||
             df == print_initset)
    { loesch_arg(&STAT_verg(*ptr));
    }
    else if (df == print_blockmove)
    { loesch_arg(&STAT_vc2(*ptr));
      loesch_arg(&STAT_verg(*ptr));
    }
    else if (df == print_blockequal ||
             df == print_debug ||
             df == print_trace)
    { loesch_arg(&STAT_vc2(*ptr));
      loesch_arg(&STAT_vc3(*ptr));
    }
    else if (df == print_add ||
             df == print_sub ||
             df == print_mul ||
             df == print_div ||
             df == print_pow ||
             df == print_mod ||
             df == print_and ||
             df == print_or ||
             df == print_eq ||
             df == print_ne ||
             df == print_lt ||
             df == print_le ||
             df == print_gt ||
             df == print_ge ||
             df == print_arctant ||
             df == print_strcmp ||
             df == print_getpixel ||
             df == print_openw ||
             df == print_openabsw)
    { loesch_arg(&STAT_verg(*ptr));
      loesch_arg(&STAT_vc1(*ptr));
      loesch_arg(&STAT_vc2(*ptr));
    }
    else if (df == print_one_disconnect)
    { loesch_arg(&STAT_out_port(*ptr));
    }
                /* Kommentartext freigeben */
    if (STAT_comm_text(*ptr)) free(STAT_comm_text(*ptr));
  }
                /* ganzes Programmarray freigeben */
  if (free_flag)
  { free(*prog); *prog = NULL; *anz = 0; }
}

/**************************************************************************
 ***             Funktion loesch_decl (rekursiv direkt und ueber loesch_ulist)
 ***
 *** gibt Speicherplatz einer DECLIST frei
 ***
 **************************************************************************/

loesch_decl(dec)
DECLIST *dec;
{ register int i;

  if (dec)
  { for (i = 0; i < dec->dcount; i++)
    { switch (dec->darray[i].art)
      { case KLAM  : loesch_decl(&DECL_klamdecl(dec->darray[i]));
                     break;
        case UNION : loesch_ulist(&DECL_ul(dec->darray[i]));
      }
    }
    if (dec->darray) free(dec->darray);
    null_dlist(dec);
  }
}

/**************************************************************************
 ***                       Funktion null_dlist
 ***
 *** besetzt '*d' komplett mit Null (dh. leere Deklaration)
 ***
 **************************************************************************/

null_dlist(d)
DECLIST *d;
{ register TYP t;
 
  d->darray = NULL;
  d->dcount = 0;
  for (t = BOOL;t <= SUMM; d->typ_zahl[t++] = 0);
}

/**************************************************************************
 ***             Funktion null_ulist
 ***
 *** besetzt eine UNION_LIST mit 0
 ***
 **************************************************************************/

null_ulist(ulist)
UNION_LIST *ulist;
{ register TYP t;

  ulist->uarray = NULL;
  ulist-> ucount = 0;
  for (t = BOOL; t <= SUMM; ulist->utyp_anz[t++] = 0);
}

/**************************************************************************
 ***             Funktion loesch_ulist (rekursiv ueber loesch_decl)
 ***
 *** gibt Speicherplatz einer UNION_LIST frei
 ***
 **************************************************************************/

loesch_ulist(ulist)
UNION_LIST *ulist;
{ register int i;

  if (ulist)
  { for (i = 0; i < ulist->ucount; i++)
    { loesch_decl(ulist->uarray + i); }
    if (ulist->uarray) free(ulist->uarray);
    null_ulist(ulist);
  }
}

/**************************************************************************
 ***                       Funktion loesch_arg
 ***
 *** gibt evtl. in Argument enthaltene Strings oder SIZE-Deklarationen frei
 ***
 **************************************************************************/

loesch_arg(aptr)
ARG *aptr;
{ if (ARG_typ(*aptr) == STR && ARG_con_wert(*aptr).inhalt.s_val)
  { free(ARG_con_wert(*aptr).inhalt.s_val); }
  else if (ARG_argsort(*aptr) & SIZ)
  { loesch_decl(ARG_sizedec(*aptr)); }
}

/**************************************************************************
 ***                       Funktion dlptr
 ***
 *** alloziert Speicherplatz fuer eine DECLIST und
 *** kopiert '*dl' dorthin.
 *** ist dl == NULL, wird eine leere Deklaration erzeugt.
 ***
 **************************************************************************/

DECLIST *dlptr(dl)
DECLIST *dl;
{ register DECLIST *erg = (DECLIST *)malloc((size_t)sizeof(DECLIST));

  if (!erg)
  { fatal(texte[66]); return NULL; }
  if (dl) *erg = *dl;
  else null_dlist(erg);
  return erg;
}

/**************************************************************************
 ***                       Funktion neu_decl
 ***
 *** erzeugt neues Listenelement fuer Deklarationsliste mit:
 *** 'art == dart' und
 *** haengt es in '*listptr' an 'darray' an.
 ***
 **************************************************************************/

DECL *neu_decl(listptr,dart)
DECLIST *listptr;
DECL_ART dart;
{ register DECL *neu;

  if (!(neu = (DECL *)(listptr->darray ?
                       realloc(listptr->darray,
                                 (size_t)(sizeof(DECL) * ++listptr->dcount)) :
                       calloc((size_t)++listptr->dcount, (size_t)sizeof(DECL)))))
  { if (listptr->darray) loesch_decl(listptr);
    fatal(texte[66]); return NULL;
  }
  listptr->darray = neu;
  neu += listptr->dcount - 1;
  neu->art = dart;
  return neu;
}

/**************************************************************************
 ***                       Funktion neu_ulist
 ***
 *** haengt '*dl' als weitere Variante in '*ulist' ein
 ***
 **************************************************************************/

neu_ulist(ulist, dl)
UNION_LIST *ulist;
DECLIST *dl;
{ register TYP t;
  register DECLIST *neu;
  register int n;

  if (!(neu = (DECLIST *)(ulist->uarray ?
                          realloc(ulist->uarray,
                                  (size_t)(sizeof(DECLIST) * ++ulist->ucount)) :
                          calloc((size_t)++ulist->ucount, (size_t)sizeof(DECLIST)))))
  { if (ulist->uarray) loesch_ulist(ulist);
    fatal(texte[66]); return;
  }
  ulist->uarray = neu;
  neu[ulist->ucount - 1] = *dl;
  for (t = BOOL; t < SUMM; ulist->utyp_anz[t] += dl->typ_zahl[t], t++);
  if ((n = dl->typ_zahl[SUMM]) > ulist->utyp_anz[SUMM])
    ulist->utyp_anz[SUMM] = n;
}

/**************************************************************************
 ***                       Funktion nextstat
 ***
 *** setzt den aktuellen Befehl auf den naechsten Platz im Programmspeicher
 *** erweitert den Programmspeicher bei Bedarf
 ***
 **************************************************************************/

nextstat()
{ nach_end++;
  if (++stat_anz > akt_max)
#ifdef MAC
#  define UMAC
#else 
#  ifndef PC
#    define UMAC
#  endif
#endif

#ifdef UMAC
  { STAT *neu;

    if (!(neu = (STAT *)realloc(programm,
                                (size_t)(((akt_max += STARTPROGLEN) + 1) *
                                         sizeof(STAT)))))
    { if (!(neu = (STAT *)realloc(programm,
                                  (size_t)(((akt_max = stat_anz) + 1) * sizeof(STAT)))))
      { stat_anz--; fatal(texte[60]); return; }
    }
    programm = neu;
#else
#define SEGMAX (((size_t)1 << 16) - 1)
  { if (akt_max >= (int)(SEGMAX / (size_t)sizeof(STAT) -1))
    { stat_anz--; fatal(texte[60]); return; }
    if ( (long)(akt_max + STARTPROGLEN + 1) * (long)sizeof(STAT) >= (long)SEGMAX )
    { if (!(programm =
             (STAT *)realloc(programm,
                             (size_t)(((akt_max = (int)(SEGMAX / sizeof(STAT) -1)) + 1) *
                                      sizeof(STAT)))))
      { stat_anz = 0; fatal(texte[60]); return; }
    }
    else
    { if (!(programm = (STAT *)realloc(programm,
                                       (size_t)(((akt_max += STARTPROGLEN) + 1) *
                                                sizeof(STAT)))))
      { stat_anz = 0; fatal(texte[60]); return; }
    }
#endif
    stat_ptr = &programm[stat_anz];
    memset(stat_ptr, 0, (size_t)(&programm[akt_max + 1] - stat_ptr) *
                        (size_t)sizeof(STAT));
  }
  else
  { stat_ptr++; }
  STAT_label(*stat_ptr) = -1;
}

/**************************************************************************
 ***                       Funktion operation
 ***
 *** erzeugt als neuen Befehl die Operation 'erg := a1 op a2'
 *** ('op' ist Token eines arithmetischen Operators)
 ***
 **************************************************************************/

operation(erg,a1,op,a2)
ARG *erg, *a1, *a2;
int op;
{ register int v_s = sign(ARG_argsort(*erg) & VEC);

  funcs(op_funcs[op-PLUS].do_f[v_s][ARG_typ(*erg)], op_funcs[op-PLUS].print_f);
  STAT_vc1(*stat_ptr) = *a1;
  STAT_vc2(*stat_ptr) = *a2;
  STAT_verg(*stat_ptr) = *erg;
}

/**************************************************************************
 ***                       Funktion rel_operation
 ***
 *** erzeugt als neuen Befehl die Operation 'erg := a1 rop a2'
 *** ('rop' ist Token eines Vergleichsoperators)
 ***
 **************************************************************************/

rel_operation(erg,a1,rop,a2)
ARG *erg, *a1, *a2;
int rop;
{ register int v_s = sign(ARG_argsort(*erg) & VEC);

  funcs(rel_op_funcs[rop-EQ].do_rf[v_s], rel_op_funcs[rop-EQ].print_rf);
  STAT_vc1(*stat_ptr) = *a1;
  STAT_vc2(*stat_ptr) = *a2;
  STAT_verg(*stat_ptr) = *erg;
}

/**************************************************************************
 ***                       Funktion funcs
 ***
 *** traegt do- und print-Funktion im aktuellen Befehl ein
 ***
 **************************************************************************/

funcs(do_fu, print_fu)
int (* do_fu)(), (* print_fu)();
{ STAT_do_func(*stat_ptr) = do_fu;
  STAT_print_func(*stat_ptr) = print_fu;
}

/**************************************************************************
 ***                       Funktion ska_vec_test
 ***
 *** testet ob '*quell' an '*ziel' zuweisbar ist (bzgl. SCALAR/VECTOR-Eigenschaft)
 ***
 **************************************************************************/

ska_vek_test(ziel,quell)
ARG *ziel,*quell;
{ if (!(ARG_argsort(*ziel) & VEC) &&
      (ARG_argsort(*quell) & VEC))
  { sem_error(texte[34]);
    fehler(texte[67]);
    ARG_argsort(*quell) |= FALSCH;
  }
}

/**************************************************************************
 ***                       Funktion wert_typ_set
 ***
 *** Ermittelt die Menge, die den Typ des Werts eines Arguments
 *** mit 'argsort == asort' enthaelt
 *** (INT bei Adressierung, sonst Typ des Arguments selbst)
 ***
 **************************************************************************/

int wert_typ_set(asort)
ARGART asort;
{ if (asort & (ADS | ADV)) return (1<<INT);
  return (1<<AA_typ(asort));
}

/**************************************************************************
 ***                       Funktion typ1_test
 ***
 *** testet '*a' auf Vertraeglichkeit mit der Typmenge 'typs'
 ***
 **************************************************************************/

typ1_test(a,typs)
ARG *a;
int typs;
{ if (!(wert_typ_set(ARG_argsort(*a)) & typs))
  { sem_error(texte[28]);
    fehler(texte[68]);
    ARG_argsort(*a) |= FALSCH;
  }
}

/**************************************************************************
 ***                       Funktion typ2_test
 ***
 *** testet '*a1' und '*a2' auf Typgleichheit und auf Vertraeglichkeit
 *** mit der Typmenge 'typs'
 ***
 **************************************************************************/

typ2_test(a1,a2,typs)
ARG *a1, *a2;
int typs;
{ register int t1, t2;
  register int ok1 = TRUE;
  register int ok2 = TRUE;

  if (!((t1 = wert_typ_set(ARG_argsort(*a1))) & typs))
  { sem_error(texte[28]);
    fehler(texte[68]);
    ARG_argsort(*a1) |= FALSCH;
    ok1 = FALSE;
  }
  if (!((t2 = wert_typ_set(ARG_argsort(*a2))) & typs))
  { sem_error(texte[28]);
    fehler(texte[68]);
    ARG_argsort(*a2) |= FALSCH;
    ok2 = FALSE;
  }
  if (!(t1 & t2) && ok1 && ok2)
  { sem_error(texte[28]);
    fehler(texte[69]);
    ARG_argsort(*a2) |= FALSCH;
  }
}

/**************************************************************************
 ***                       Funktion typ3_test
 ***
 *** testet '*a1', '*a2' und '*a3' auf Typgleichheit und auf Vertraeglichkeit
 *** mit der Typmenge 'typs'
 ***
 **************************************************************************/

typ3_test(a1,a2,a3,typs)
ARG *a1, *a2, *a3;
int typs;
{ register int t1, t2, t3;
  register int ok1 = TRUE;
  register int ok2 = TRUE;
  register int ok3 = TRUE;

  if (!((t1 = wert_typ_set(ARG_argsort(*a1))) & typs))
  { sem_error(texte[28]);
    fehler(texte[68]);
    ARG_argsort(*a1) |= FALSCH;
    ok1 = FALSE;
  }
  if (!((t2 = wert_typ_set(ARG_argsort(*a2))) & typs))
  { sem_error(texte[28]);
    fehler(texte[68]);
    ARG_argsort(*a2) |= FALSCH;
    ok2 = FALSE;
  }
  if (!((t3 = wert_typ_set(ARG_argsort(*a3))) & typs))
  { sem_error(texte[28]);
    fehler(texte[68]);
    ARG_argsort(*a3) |= FALSCH;
    ok3 = FALSE;
  }
  if (!(t1 & t2) && ok1 && ok2)
  { sem_error(texte[28]);
    fehler(texte[69]);
    ARG_argsort(*a2) |= FALSCH;
  }
  if (!(t1 & t3) && ok1 && ok3)
  { sem_error(texte[28]);
    fehler(texte[69]);
    ARG_argsort(*a3) |= FALSCH;
  }
}

/**************************************************************************
 ***                       Funktion port_nr_test
 ***
 *** ueberprueft ob das Argument '*aptr' als Portnummer verwendet werden kann
 ***
 **************************************************************************/

port_nr_test(aptr)
ARG *aptr;
{ register int i;

  if (ARG_typ(*aptr) != INT)            /* Typ muss INTEGER sein */
  { sem_error(texte[28]);
    fehler(texte[70]);
    ARG_argsort(*aptr) |= FALSCH;
  }
  else if (ARG_argsort(*aptr) & (VEC | ADS | ADV | SIZ))/* keine Skalarvariable */
  { sem_error(texte[57]);
    fehler(texte[71]);
    ARG_argsort(*aptr) |= FALSCH;
  }
  else if ((ARG_argsort(*aptr) & CON) &&
           ((i = ARG_con_wert(*aptr).inhalt.i_val) < 1 || i > port_anz))
                                /* falscher Konstantenwert */
  { sem_error(texte[64],i); fehler_zahl++;
    ARG_argsort(*aptr) |= FALSCH;
  }
}

/**************************************************************************
 ***                       Funktion end_test
 ***
 *** testet ob - END-Befehl existiert und letzter Befehl ist,
 ***           - jedes aufgerufene Label existiert,
 ***           - jeder geCALLte Befehl ein PROC ist,
 ***           - kein GOTO auf PROC-Befehl geht,
 *** setzt Befehlsnummern fuer Labels ein,
 *** begrenzt Programmspeicher auf benoetigte Groesse,
 *** ermittelt SYSTEM-Namen im END-Kommentar.
 ***
 **************************************************************************/

end_test()
{ int z, n;
  STAT *sptr;

  if (nach_end == stat_anz)     /* kein END : erzeugen */
  { sem_warning(texte[74]);
    funcs(do_end, print_end);
    STAT_label(*stat_ptr) = -1;
    nextstat();
    if (abbruch) return;
  }
  else if (nach_end > 0)        /* END nicht letzter Befehl */
  { sem_error(texte[72]);
    fehler(texte[75]);
    STAT_falsch(*(stat_ptr - nach_end - 1)) = TRUE;
  }
                /* Sprungziele bei CALL, GOTO und REDUCE untersuchen */
  for (z = 0,sptr = programm; z < stat_anz; z++,sptr++)
                /* CALL-Befehle */
  { if (STAT_print_func(*sptr) == print_ifcall ||
        STAT_print_func(*sptr) == print_whilecall ||
        STAT_print_func(*sptr) == print_call)
    { if ((n = STAT_spr_ziel(*sptr)) >= 0)      /* Ziel schon bekannt */
      { if (STAT_print_func(programm[n]) != print_proc)    /* Fehler schon gemeldet */
        { STAT_spr_ziel(*sptr) = -STAT_label(programm[n]); }    /* Label falsch */
      }
      else
      { STAT_spr_ziel(*sptr) = n = ziel(n);
        if (n < 0)              /* Label existiert nicht */
        { sem_error(texte[72]);
          fehler(texte[76], -n, STAT_label(*sptr));
        }
        else if (STAT_print_func(programm[n]) != print_proc) /* Ziel kein PROC */
        { sem_error(texte[72]);
          fehler(texte[77], n = STAT_label(programm[n]), STAT_label(*sptr));
          STAT_spr_ziel(*sptr) = -n;
        }
      }
    }
                /* GOTO-Befehle */
    else if (STAT_print_func(*sptr) == print_ifgoto ||
             STAT_print_func(*sptr) == print_goto)
    { if ((n = STAT_spr_ziel(*sptr)) >= 0)      /* Ziel schon bekannt */
      { if (STAT_print_func(programm[n]) == print_proc)    /* Fehler schon gemeldet */
        { STAT_spr_ziel(*sptr) = -STAT_label(programm[n]); }    /* Label falsch */
      }
      else
      { STAT_spr_ziel(*sptr) = n = ziel(n);
        if (n < 0)      /* Label existiert nicht */
        { sem_error(texte[72]);
          fehler(texte[76], -n, STAT_label(*sptr));
        }
        else if (STAT_print_func(programm[n]) == print_proc)    /* Ziel ist PROC */
        { sem_error(texte[72]);
          fehler(texte[73], n = STAT_label(programm[n]), STAT_label(*sptr));
          STAT_spr_ziel(*sptr) = -n;
        }
      }
    }
                /* REDUCE-Befehle */
    else if (STAT_print_func(*sptr) == print_proc_reduce ||
             STAT_print_func(*sptr) == print_redproc_propagate ||
             STAT_print_func(*sptr) == print_redproc_send ||
             STAT_print_func(*sptr) == print_redproc_receive)
    { if ((n = STAT_red_fct(*sptr)) >= 0)       /* Ziel schon bekannt */
      { if (STAT_print_func(programm[n]) != print_proc)    /* Fehler schon gemeldet */
        { STAT_red_fct(*sptr) = -STAT_label(programm[n]); }     /* Label falsch */
      }
      else
      { STAT_red_fct(*sptr) = n = ziel(n);
        if (n < 0)      /* Label existiert nicht */
        { sem_error(texte[72]);
          fehler(texte[76], -n, STAT_label(*sptr));
        }
        else if (STAT_print_func(programm[n]) != print_proc)    /* Ziel nicht PROC */
        { sem_error(texte[72]);
          fehler(texte[78], n = STAT_label(programm[n]), STAT_label(*sptr));
          STAT_red_fct(*sptr) = -n;
        }
      }
    }
  }
  programm = (STAT *)realloc(programm,(size_t)(stat_anz * sizeof(STAT)));
  akt_max = stat_anz - 1;
}

#ifndef CRAY
/**************************************************************************
 ***                       Funktion parse_quit_func
 ***
 *** wird aufgerufen, wenn Parser von aussen unterbrochen wird:
 *** unterbindet weitere Fehlermeldungen und Warnungen
 ***
 **************************************************************************/

void parse_quit_func(sig)
int sig;
{ signal(sig, parse_quit_func);
  if (anz_meldungen < fehlermax) anz_meldungen = fehlermax - 1;
}
#endif

/**************************************************************************
 ***            Funktion parse
 ***
 *** entfernt Reste frueherer Programmlaeufe,
 *** startet Parser fuer PARZ-Programm.
 *** 'pip == TRUE' : Eingabedatei ist Pipe
 *** Ergebnis : TRUE, wenn keine Fehler auftraten
 ***
 **************************************************************************/

int parse(pip)
int pip;
{ int (* altesigs[3])();

#ifndef CRAY
  sigon(altesigs, parse_quit_func);     /* Unterbrechung abfangen */
#endif
  aufraeumen();
  if (portarray)
  { free(portarray); portarray = NULL; }
  flags = 0;
  yynerrs = 0;
  anz_meldungen = anz_fehler = 0;
  fehler_zahl = 1;
  ausgeben = TRUE;
  if (vec_stacks)
  { free(vec_stacks); vec_stacks = NULL; }
  loesch_decl(&glob_s_decl);
  loesch_decl(&glob_v_decl);
  if (programm) loesch_prog(&programm, &stat_anz, TRUE);
  parse_start = 1;              /* Parser fuer Programm */
  unputc('$');                  /* wird zu PROGRAMMSTART */
  abbruch = FALSE;
  yyerfp = stderr;
  yylineno = 1;
  max_source_zeile = max_label = 0;
  yyparse();
  stepping_to = forced_break = FALSE;
  ss_modus = (max_source_zeile ? STEP_COMP : STEP_INTER);
  if (ss_mod_start) ss_mod_start = ss_modus; 
  ss_mod = ss_mod_start;
  szlen = max_source_zeile ? (int)log10((double)max_source_zeile) + 1 : 0;
  lablen = max_label ? (int)log10((double)max_label) + 1 : 1;
#  ifndef PC
  if (pip) 
  { if (pclose(yyin)) anz_fehler++; }
  else
#  endif
       if (yyin != stdin) fclose(yyin);
#ifndef CRAY
  sigoff(altesigs);                     /* Signale zuruecksetzen */
#endif
  return (anz_fehler == 0 && !abbruch);
}

#ifndef CRAY
/**************************************************************************
 ***         Funktion kommandoparse
 ***
 *** startet Parser fuer Kommandos
 *** Ergebnis : TRUE, wenn keine Fehler autraten
 ***
 **************************************************************************/

int kommandoparse()
{ yyin = kommandin;
  yynerrs = 0;
  anz_meldungen = anz_fehler = flags = 0;
  fehler_zahl = 1;
  ausgeben = TRUE;
  parse_start = 2;      /* Parser fuer Kommandos */
  unputc('$');          /* wird zu KOMMANDOSTART */
  abbruch = FALSE;
  yyerfp = (kommandout == stdout) ? stderr : kommandout;
  return yyparse();
}

/**************************************************************************
 ***                       Funktion show_prompt
 ***
 *** gibt Prompt fuer Kommandoeingabe aus,
 *** fuehrt Initialisierungen fuer neues Kommando durch
 ***
 **************************************************************************/

show_prompt()
{ kill_debcode();
  if (no_linefeed && kommandout == stdout) putc('\n', kommandout);
  putc('\r', kommandout);
  fflush(kommandout);
  no_linefeed = FALSE;
  fputs(promptstr[breaked], yyerfp);
  fflush(yyerfp);
  anz_meldungen = anz_fehler = quitted = more_count = 0;
  err = fperror = FALSE;
  lesend = nospez = FALSE;
  parse_start = -2;
  para_off();
  bicount = -1;         /* Argumentzaehler fuer 'item_ptr()' */
  ausgeben = bef_lesen = TRUE;
  scanner_fehler = FALSE;
}

/**************************************************************************
 ***                       Funktion beende_parser
 ***
 *** hinterlaesst ordentlichen Eingabestrom wenn Kommandoparser zwecks
 *** Simulation verlassen wird.
 ***
 **************************************************************************/

beende_parser()
{ extern int yychar;

  if (yychar > 0)
  { ups();
    ueberlies_zeile();
  }
}

#endif
