(*
    Author:	Sven Schiller
    Course:	ICSS 590
    Purpose:	Implementation of the "Game of Life" in a parallel computing
		environment (SIMD).
    Description:
	The total size of the simulation is limited to 10x10 cells. A 
	mesh topology was utilized. The number of processing elements are
	100 (10*10). Each PE is connected to its 8 neighbors in order to
	allow the PEs to pass cell information to all of their neighbors.

	The algorithm is as follows:
	Each cell passes the life information for itself to its neighbors
	(each generation 1 time to a neighbor in each direction, or 8 times).
	The cell itself evaluates the information passed to it by its 
	neighbors, that is, it increments a counter for the number of
	neighbor each time it finds that its neighbor is alive.
	After the number of neighbors has been determined, the cell decides
	what to do with itself (be born, or die).

    Created:	Mar 16, 1991
*)

SYSTEM Game_of_Life;

CONST
    MAXHOR = 10;	(* maximum horizontal no. of cells *)
    MAXVER = 10;	(* maximum vertical no. of cells *)

TYPE
    CellArrType = ARRAY[0..MAXVER-1],[0..MAXHOR-1] OF BOOLEAN; (* cell array *)

CONFIGURATION
    cells [0..MAXVER-1], [0..MAXHOR-1];	(* mesh of cells: y,x *)

CONNECTION   (* each cell is connected to each of its 8 neighbors *) 
    up		: cells[y,x] -> cells[y+1,x].down;	  (* up connection *)
    down	: cells[y,x] -> cells[y-1,x].up;	  (* down connection *)
    right	: cells[y,x] -> cells[y,x+1].left;	  (* right connection *)
    left	: cells[y,x] -> cells[y,x-1].right;	  (* left connection *)
    diagR_up	: cells[y,x] -> cells[y+1,x+1].diagL_down;(* diag. right-up *)
    diagR_down	: cells[y,x] -> cells[y-1,x+1].diagL_up;  (* diag. right-down *)
    diagL_up	: cells[y,x] -> cells[y+1,x-1].diagR_down;(* diag. left-up *)
    diagL_down	: cells[y,x] -> cells[y-1,x-1].diagR_up;  (* diag. left-down *)

SCALAR
    liveCells	: CARDINAL;	(* number of live cells *)
    generation  : CARDINAL;	(* current generation *)
    maxGens	: CARDINAL;	(* maximum number of generations *)
    birthCond	: CARDINAL;	(* number of neighbors to allow birth *)
    isoCond	: CARDINAL;	(* number of neighb. to kill from isolation *)
    overCond	: CARDINAL;	(* number of neighb. to kill from crowding *)
    births	: CARDINAL;	(* number of births (last generation) *)
    deaths	: CARDINAL;	(* number of deaths (last generation) *)
    living	: CellArrType;	(* contains life inform. for all cells *)

VECTOR
    alive	: BOOLEAN;	(* life status of cell *)
    died	: CARDINAL;	(* 1=this cell just died, 0=otherwise *)
    born	: CARDINAL;	(* 1=this cell was just born, 0=otherwise *)

(*----------------------------------------------------------------------------
    Name: 	NeighborNum
    Purpose:	Determine the number of live neighbors for each processor.
    Parameters: 
	Input:
	    alive	- flag for alive cell
	Output:
	    neighbors	- number of alive neighbors
----------------------------------------------------------------------------*)
PROCEDURE NeighborNum(VECTOR alive : BOOLEAN; VECTOR VAR neighbors : CARDINAL);
VECTOR
    neighLife	: BOOLEAN;	(* alive value from neighbor *)
BEGIN
    neighbors := 0;
    neighLife := FALSE;
    PROPAGATE.up (alive, neighLife);
    IF (dim1 # 0) AND neighLife THEN	(* do not enter statement for PE's  *)
	INC(neighbors);			(* with y=0 (no input port for UP). *)
    END; (* IF *)
    neighLife := FALSE;
    PROPAGATE.down (alive, neighLife);
    IF (dim1 # MAXVER-1) AND neighLife THEN
	INC(neighbors);
    END; (* IF *)
    neighLife := FALSE;
    PROPAGATE.left (alive, neighLife);
    IF (dim2 # MAXHOR-1) AND neighLife THEN
	INC(neighbors);
    END; (* IF *)
    neighLife := FALSE;
    PROPAGATE.right (alive, neighLife);
    IF (dim2 # 0) AND neighLife THEN
	INC(neighbors);
    END; (* IF *)
    neighLife := FALSE;
    PROPAGATE.diagL_up (alive, neighLife);
    IF (dim1 # 0) AND (dim2 # MAXHOR-1) AND neighLife THEN
	INC(neighbors);
    END; (* IF *)
    neighLife := FALSE;
    PROPAGATE.diagL_down (alive, neighLife);
    IF (dim1 # MAXVER-1) AND (dim2 # MAXHOR-1) AND neighLife THEN
	INC(neighbors);
    END; (* IF *)
    neighLife := FALSE;
    PROPAGATE.diagR_up (alive, neighLife);
    IF (dim1 # 0) AND (dim2 # 0) AND neighLife THEN
	INC(neighbors);
    END; (* IF *)
    neighLife := FALSE;
    PROPAGATE.diagR_down (alive, neighLife);
    IF (dim1 # MAXVER-1) AND (dim2 # 0) AND neighLife THEN
	INC(neighbors);
    END; (* IF *)
END NeighborNum;

(*----------------------------------------------------------------------------
    Name: 	DoGeneration
    Purpose:	Changes life values in cells to generate the next generation.
    Parameters: 
	InOut:
	    alive	- flag for alive cell
	Output:
	    died	- 1=cell died in this generation, 0 otherwise
	    born	- 1=cell was born in this generation, 0 otherwise
----------------------------------------------------------------------------*)
PROCEDURE DoGeneration(VECTOR VAR alive : BOOLEAN; 
		       VECTOR VAR died, born : CARDINAL);
VECTOR
    neighbors	: CARDINAL;	(* no. of alive neighbors *)
BEGIN
    NeighborNum(alive, neighbors);
    born := 0;
    died := 0;
    IF NOT alive AND (neighbors = birthCond) THEN
	alive := TRUE;
	born := 1;
    ELSIF alive AND (neighbors < isoCond) THEN
	alive := FALSE;
	died := 1;
    ELSIF alive AND (neighbors > overCond) THEN
	alive := FALSE;
	died := 1;
    END; (* IF *)
END DoGeneration;

(*----------------------------------------------------------------------------
    Name: 	Input
    Purpose:	Reads in the initial data from stdin. Input format is assumed
		correct.
    Parameters: 
	Output:
	    liveCells	- number of cells alive initially
	    maxgens	- maximum number of generations
	    birthCond	- condition for birth in a cell
	    isoCond	- condition for death because of isolation
	    overCond	- condition for death because of overcrowding
	    living	- array with flags for life condition of cells
----------------------------------------------------------------------------*)
PROCEDURE Input(SCALAR VAR liveCells, maxgens, birthCond, isoCond,
		overCond : CARDINAL; SCALAR VAR living : CellArrType);
SCALAR
    inNum	: CARDINAL;	(* number read in *)
    x,y		: CARDINAL;	(* loop control variables *)
BEGIN
    liveCells := 0;
    FOR y := MAXVER-1 TO 0 BY -1 DO
	FOR x := 0 TO MAXHOR-1 DO
	    ReadCard(inNum);
	    living[y,x] := (inNum=1);
	    liveCells := liveCells + inNum;
	END; (*FOR *)
    END; (* FOR *)
    ReadCard(birthCond);
    ReadCard(isoCond);
    ReadCard(overCond);
    ReadCard(maxgens);
END Input;


(*----------------------------------------------------------------------------
    Name: 	Output
    Purpose:	Write information about generation to stdout. Output includes
		the generation number, the number of births and deaths since
		the last generation, and a matrix containing all cells
		represented by 1 or 0 (for alive or dead).
    Parameters: 
	Input:
	    liveCells	- number of cells alive initially
	    generation	- number of generation
	    deaths	- number of deaths since last generation
	    births	- number of births since last generation
	    living	- life inform. for all cells
----------------------------------------------------------------------------*)
PROCEDURE Output(SCALAR liveCells, generation, deaths, births : CARDINAL;
		 SCALAR living : CellArrType); 
SCALAR
    x,y		: CARDINAL;	(* loop control variables *)
BEGIN
    WriteString("Generation: ");
    WriteCard(generation, 2);
    WriteLn;
    WriteString("   Number of Births: ");
    WriteCard(births, 2);
    WriteLn;
    WriteString("   Number of Deaths: ");
    WriteCard(deaths, 2);
    WriteLn;
    WriteString("   Total Population: ");
    WriteCard(liveCells, 2);
    WriteLn;
    WriteLn;
    FOR y := MAXVER-1 TO 0 BY -1 DO
	FOR x := 0 TO MAXHOR-1 DO
	    IF living[y,x] THEN
		WriteString("1 ");
	    ELSE
		WriteString("0 ");
	    END; (* IF *)
	END; (* FOR *)
	WriteLn;
    END; (* FOR *)
    WriteLn;
    WriteLn;
END Output;


(* 
 * MAIN BODY: Initializes some variables, reads the input, and then
 *	      controls the simulation. Also outputs information about
 *	      every generation.
 *)

BEGIN
    deaths := 0;
    births := 0;
    openinput("lab1.dat");
    Input(LiveCells, maxGens, birthCond, isoCond, overCond, living);
    closeinput;
    LOAD(alive, living);
    FOR generation := 0 TO maxGens DO
	Output(liveCells, generation, deaths, births, living);
	PARALLEL
	    DoGeneration(alive, died, born);
	ENDPARALLEL;
	STORE(alive, living);
	deaths := REDUCE.SUM(died);
	births := REDUCE.SUM(born);
	liveCells := liveCells - deaths + births;
    END; (* FOR *)
END Game_of_Life.
