/*****************************************************************************
  Project: PARZ - Parallel Intermediate Code Debugger/Interpreter
  ----------------------------------------------------------------------------
  Release      : 1
  Project Part : X - Interface
  Filename     : xgraph.c       
  SCCS-Path    : /tmp_mnt/user/sembach/parz/v2/SCCS/s.xgraph.c
  Version      : 1.3 
  Last changed : 9/27/93 at 14:05:08        
  Author       : Heiner Braun und Michael Brenner (Mod. by Frank Sembach)
  Creation date: 01.10.91
  ----------------------------------------------------------------------------
  Description  : Einfache Grafikschnittstelle fuer XWindows 

******************************************************************************
***      (C) COPYRIGHT University of Stuttgart - All Right Reserved        ***
*****************************************************************************/

static char sccs_id[] = "@(#)xgraph.c	1.3  9/27/93 PARZ - X - Interface (Heiner Braun, Michael Brenner)";


/*****************************************************************************/
/* Programm      : XInterface.c                                              */
/* Authoren      : Heiner Braun und Michael Brenner                          */
/* Datum         : 01.10.91                                                  */
/* Beschreibung  : Einfache Grafikschnittstelle fuer XWindows                */
/*****************************************************************************/

#include <stdio.h>

#include "X11/Xlib.h"
#include "X11/Xutil.h"

#define GRAPHC

#ifdef PZ2C
#include "pz2c.h"
#endif

#ifdef PZ2MPLS
#include "pz2mpls.h"
#endif

#ifdef PZ2MPL
#include "pz2mpl-fe.h"
#endif

#ifdef PARZ
#include "parzdefs.h"
#include "externs.h"
#include "rundefs.h"
#include "runexts.h"
#include "xgraph.h"
#endif

#define PXOK 0
#define PXWRONGWINDOWSIZE 1
#define PXMEMORYTROUBLE 2
#define PXWINDOWCREATIONFAILURE 3
#define PXNOTEXISTINGWINDOWNUMBER 4
#define PXNOWINDOWSELECTED 5
#define PXWRONGCOORDINATES 6
#define PXWRONGCOLORS 7

#define FONT_ENV_VAR   "PARALLAXIS_FONT"

char *font_name;

XFontStruct  *font;


/*****************************************************************************/
/* X-spezifische Variablen */

static Display               *display;
static GC                    gc;
static Screen                *screen;
static int                   screen_num,
                             screenwidth,
                             screenheight,
                             default_depth;
static Visual                *default_visual;
static Window                root;
static Colormap              color_map;

/*****************************************************************************/


struct WindowList {    struct WindowList   *nextW;
                       Window              wPointer;
                       int                 winNum,
                                           width, height,
                                           pixel,
		                          *pixelList,
		                           pixelListSize,
                                           x, y;
                  };

int px_error;

static struct WindowList   *firstW = NULL,
                           *selectedW = NULL,
                           *lastW = NULL,
                            maxW;


#ifdef PZ2MPLS

char *New (amount)

  unsigned amount;

{
   char *ret;

   if ((ret = (char *) malloc (amount)) == NULL) {
     fprintf (stderr, "Can't allocate memeory. Sorry.\n");
     exit (1);
   }
 
   return (ret);
}

#endif

/*****************************************************************************/

int init_x()

{
  char *displayName = NULL;

  if (displayName = (char *) getenv ("DISPLAY"))
  {
    if( !(display = XOpenDisplay(displayName)) )
    {
      fprintf (stderr, "Can't open display '%s'\n", displayName);
      px_error = PXWINDOWCREATIONFAILURE;
      return( -1 );
    }
  }
  else
  { fprintf (stderr, "$DISPLAY not set\n");
    px_error = PXWINDOWCREATIONFAILURE;
    return( -1 );
  }

  screen = XDefaultScreenOfDisplay( display );
  screen_num = XDefaultScreen( display );
  root = XRootWindowOfScreen( screen );
  screenwidth = XWidthOfScreen( screen );
  screenheight = XHeightOfScreen( screen );
  default_depth = XDefaultDepthOfScreen( screen );
  default_visual = XDefaultVisualOfScreen( screen );
  color_map = XDefaultColormapOfScreen( screen );
  gc = XDefaultGC( display, screen_num );

  /* If other font than default-font is wanted, set actual font to wanted font */
  if ((font_name = (char *) getenv (FONT_ENV_VAR)) != NULL) {
    font = (XFontStruct *) XLoadFont (display, font_name);
    XSetFont (display, gc, font);
  }

  maxW.width  = screenwidth;
  maxW.height = screenheight;

  return( 1 );
}

/*****************************************************************************/

int OpenW( xSize, ySize,peNum )
float xSize, ySize;
int   peNum;

{ return OpenAbsW((int)(screenwidth * xSize), (int)(screenheight * ySize), peNum);
}

/*****************************************************************************/

int OpenAbsW( xSize, ySize,peNum )
int xSize, ySize;
int   peNum;

{
  char                         WName[25];
  struct WindowList            *newW;
  XSetWindowAttributes         xswa;
  XSizeHints                   xsh;

  px_error = PXOK;

  /* Window Groesse OK ? */

  if( xSize > screenwidth || xSize <= 0 || ySize > screenheight || ySize <= 0 )
  {
    px_error = PXWRONGWINDOWSIZE;
    return( -1 );
  }

  /* Speicherplatz fuer neuen Struct */

  if( (newW = (struct WindowList *) malloc(sizeof(struct WindowList))) == NULL)
  {
    px_error = PXMEMORYTROUBLE;
    return( -2 );
  }

  if ((newW->pixelList = (int *)malloc(sizeof(int) * ((peNum < 1) ? 1 : peNum)))
      == NULL)
    {
      px_error = PXMEMORYTROUBLE;
      return( -2 );
    }

  newW -> pixelListSize = peNum;

  /* Ist schon ein Displaygeoeffnet ??? */

  if( display == NULL )
  {
    if( init_x() < 0 )  /* Initialisierung mit Allem */
      return( -3 );
  }

  /* Vergabe der eindeutigen Windownummer fuer folgende Fkt's */

  if( lastW )
    newW->winNum = lastW->winNum + 1;
  else
    newW->winNum = 1;
  
  /***** Belegung der Variablen des neuen Struct's *****/

  newW->nextW = NULL;

  newW->width = xSize;
  newW->height = ySize;

  newW->pixel = XBlackPixelOfScreen( screen );

  newW->x = 0;
  newW->y = 0;

  /* Windowattribute */

  xswa.event_mask = ButtonPressMask;            
  xswa.background_pixel = XWhitePixelOfScreen( screen );
  xswa.bit_gravity = NorthWestGravity;
  xswa.backing_store = Always;

  newW->wPointer = XCreateWindow( display, root, 0, 0, newW->width,
                                  newW->height, 5, default_depth, InputOutput,
                                  default_visual, CWEventMask | CWBackPixel |
                                  CWBitGravity | CWBackingStore, &xswa );

  /* Colormap fuer neues Window */

  XSetWindowColormap( display, newW->wPointer, color_map );

/***** Setzen der Zeiger fuer die Liste *****/

  selectedW = newW;

  if( lastW )
  {
    lastW->nextW = newW;
    lastW = newW;
  }
  else
    firstW = lastW = newW;

  /* Window- und Iconname gleich der winNum */

  sprintf (WName, "Parallaxis Window %d", newW -> winNum);

  XStoreName( display, newW->wPointer, WName );
  XSetIconName( display, newW->wPointer, WName );


  /* Hier wird Resize verhindert */

  xsh.flags = PPosition | PSize | PMinSize | PMaxSize;

  xsh.min_width = xsh.width = xsh.max_width = newW->width;
  xsh.min_height = xsh.height = xsh.max_height = newW->height;

  XSetNormalHints( display, newW->wPointer, &xsh );

  /* Window wird angezeigt */

  XMapWindow( display, newW->wPointer ); /* Window auf den Bildschirm bringen*/
  XFlush (display);

  return( newW->winNum ); /* Windownummer */
}

/*****************************************************************************/

int CloseW( wNum )
int wNum;

{
  struct WindowList       *help, *oldhelp;

  px_error = PXOK;

  for( help = firstW, oldhelp = NULL; (help != lastW) && (help->winNum < wNum); oldhelp = help, help = help->nextW );

  if( help && (help->winNum == wNum) ) /* stimmt Windownummer */
  {
    if( oldhelp )                      /* Ausschluss des    */
      oldhelp->nextW = help->nextW;    /*                   */
                                       /*                   */
    if( help == firstW )               /* zu schliessenden  */
      firstW = help->nextW;            /*                   */
                                       /*                   */
    if( help == selectedW )            /* Fensters aus der  */
      selectedW = NULL;                /*                   */
                                       /*                   */
    if( help == lastW )                /* verketteten Liste */
      lastW = oldhelp;

    XDestroyWindow( display, help->wPointer ); /* Hier wird Fenster geloescht*/
/*#if 0*/
XFlush (display);
/*#endif*/
    free( help );        /* Speicherplatzfreigabe des Struct's */
  }
  else
    px_error = PXNOTEXISTINGWINDOWNUMBER;

  return (!px_error);
}

/*****************************************************************************/

int SelectW( wNum )
int wNum;

{
  struct WindowList       *help;

  px_error = PXOK;
  
  if (wNum == 0 && display)
    selectedW = &maxW;
  else {
    for( help = firstW; (help != lastW) && (help->winNum < wNum);
         help = help->nextW );

    if( help && (help->winNum == wNum) ) /* Window gefunden ??? */
      selectedW = help;
    else {
      px_error = PXNOTEXISTINGWINDOWNUMBER;
      selectedW = NULL;
    }
  }

  return (!px_error);
}

/*****************************************************************************/

int WSize(width, height )
int *width, *height;

{
  px_error = PXOK;

  if (selectedW) {
    *width  = selectedW->width;
    *height = selectedW->height;
  } else
    px_error = PXNOWINDOWSELECTED;

  return (!px_error);
}


#ifdef PZ2MPLS

int WSizeFE (width, height)
long *width, *height;

{
   int w, h, ret;

   ret = WSize (&w, &h);
   copyOut (&w, width, sizeof (long));
   copyOut (&h, height, sizeof (long));
   return (ret);
}

#endif

/*****************************************************************************/


#if ( defined(PZ2C) || defined(PARZ) )

int SetColor(rgb)
 
  COLREC *rgb;

{
           XColor         color;
  register int            i, *cols;
  register long           red, green, blue;
  register unsigned long  pix;

  red   = rgb -> r.i;
  green = rgb -> g.i;
  blue  = rgb -> b.i;

  px_error = PXOK;

  if( selectedW )
  {
    if( (red<0) || (green<0) || (blue<0) ||
        (red>255) || (green>255) || (blue>255) )
      px_error = PXWRONGCOLORS;
    else
    {
      color.red = red * 257;
      color.green = green * 257;
      color.blue = blue * 257;

      XAllocColor( display, color_map, &color );

      selectedW->pixel = color.pixel;

      cols = selectedW -> pixelList;
      pix = color.pixel;
      for (i = 0; i < selectedW -> pixelListSize; i++, cols++)
        *cols = pix;
    }
  }
  else
    px_error = PXNOWINDOWSELECTED;

   return (!px_error);
}

#else

int SetColor(rgb)
 
  COLREC *rgb;

{
           XColor         color;
	   COLREC         RGB;
  register int            i, *cols;
  register long           red, green, blue;
  register unsigned long  pix;

#ifdef PZ2MPLS
  copyIn (rgb, &RGB, sizeof (COLREC));

  red   = RGB.r.i;
  green = RGB.g.i;
  blue  = RGB.b.i;
#else
  red   = rgb -> r.i; 
  green = rgb -> g.i; 
  blue  = rgb -> b.i; 
#endif

  px_error = PXOK;

  if( selectedW )
  {
    if( (red<0) || (green<0) || (blue<0) ||
        (red>255) || (green>255) || (blue>255) )
      px_error = PXWRONGCOLORS;
    else
    {
      color.red = red * 257;
      color.green = green * 257;
      color.blue = blue * 257;

      XAllocColor( display, color_map, &color );

      selectedW->pixel = color.pixel;

      cols = selectedW -> pixelList;
      pix = color.pixel;
      for (i = 0; i < selectedW -> pixelListSize; i++, cols++)
        *cols = pix;
    }
  }
  else
    px_error = PXNOWINDOWSELECTED;

   return (!px_error);
}

#endif


#ifdef PZ2C


int VSetColor (rgb, dist)

  MemItem  *rgb;
  int       dist;

{
   register int      i, *cols;
   register char    *act;
   register MemItem *RGB;
            XColor   color;
   extern   int      peNum;

   px_error = PXOK;

   if (selectedW) {
     RGB = rgb; act = activePE; cols = selectedW->pixelList;
     for (i = 0; i < peNum; i++, act++, RGB += dist, cols++)
       if (isActive ()) {
	 color.red   = ((COLREC *) RGB) -> r.i * 257;
	 color.green = ((COLREC *) RGB) -> g.i * 257;
	 color.blue  = ((COLREC *) RGB) -> b.i * 257;
	 XAllocColor (display, color_map, &color);
	 *cols = color.pixel;
       } else
         *cols = -1;
   } else
     px_error = PXNOWINDOWSELECTED;

  return (!px_error);
}

#else
#  ifdef PARZ

int VSetColor (rits, blpt, len)
ITPTRS *rits;
VARBLOCK **blpt;
int len;
{ register int *cols;
  XColor   color;
  ITEM *rit, *git, *bit;
  VARBLOCK *bl;
  
  px_error = PXOK;

  if (selectedW)
  { cols = selectedW->pixelList;
    for (pec = 0; pec < last_akt_pe; pec++, blpt += dummybs->inc, cols++)
    { rit = (* rits->n_it_func)(rits);
      if (aktive_pes[pec] == '1')
      { bl = *blpt;
	if ((rit - bl->vars) / len + bl->von + 2 > bl->bis)
	{ runerr(texte[286]); return; }
	git = rit + len; bit = git + len;
	ini_typ_test(rit->datentyp, INT, 91, 283);
	ini_typ_test(git->datentyp, INT, 91, 283);
	ini_typ_test(bit->datentyp, INT, 91, 283);
	color.red   = rit->inhalt.i_val * 257;
	color.green = git->inhalt.i_val * 257;
	color.blue  = bit->inhalt.i_val * 257;
	XAllocColor (display, color_map, &color);
	*cols = color.pixel;
      }
      else
      { *cols = -1; }
    }
  }
  else
  { px_error = PXNOWINDOWSELECTED; }
  
  return (!px_error);
}

#  else
int VSetColor (indx, rgb, nxproc, nyproc)

  int      indx;
  char    *rgb;
  int      nxproc, nyproc;

{
   register int         i, j, *cols, max;
   register COLREC     *buf;
            XColor      color;
   extern   int         peMax;
   extern   char       *New ();
#ifdef PZ2MPL
   extern   MemItem     _DONE;
#endif

   px_error = PXOK;

   /* Kopiere die Daten vom Back-End zum Front-End */
   buf = (COLREC *) New (sizeof (COLREC) * nyproc * nxproc);
   blockIn (rgb, buf, 0, 0, nxproc, nyproc, sizeof (COLREC));

   /* Verarbeite die Daten */
   if (selectedW) {
#ifdef PZ2MPL
     _DONE.b = 1;
#endif

     cols = & (selectedW -> pixelList [indx * nxproc * nyproc]);
     for (i = 0, j = 0; i < nxproc * nyproc; i++) {
       if (buf [i].r.i != -99) {
         color.red   = buf [i].r.i * 257;
         color.green = buf [i].g.i * 257;
         color.blue  = buf [i].b.i * 257;
	 XAllocColor (display, color_map, &color);
         cols [j++] = color.pixel;
       } else
	 cols [j++] = -1;
     }
   } else {
#ifdef PZ2MPL
     _DONE.b = 0;
#endif
     px_error = PXNOWINDOWSELECTED;
   }

   free (buf);
   return (!px_error);
}
#  endif
#endif


/*****************************************************************************/

int SetPixel( x, y )
int x, y;

{
  px_error = PXOK;

  if( selectedW )
  {
    if (x >= 0 && y >= 0 && x < selectedW->width && y < selectedW->height)
      {
	XSetForeground( display, gc, selectedW->pixel );
	XDrawPoint( display, selectedW->wPointer, gc, x, y );
	XFlush (display);
      }
    else
      px_error = PXWRONGCOORDINATES;
  }
  else
    px_error = PXNOWINDOWSELECTED;


  return (!px_error);
}


#ifdef PZ2C


int VSetPixel (x, y, distX, distY)

  MemItem *x, *y;
  int      distX, distY;

{
   register int   i, *cols;
   register char *act;
   register MemItem *xx, *yy;
   extern   int   peNum;

   px_error = PXOK;

   if (selectedW) {
     xx = x;  yy = y; act = activePE; cols = selectedW->pixelList;
     for (i = 0; i < peNum; i++, act++, xx += distX, yy += distY, cols++)
       if (isActive ()) {
         XSetForeground( display, gc, *cols);
         XDrawPoint( display, selectedW->wPointer, gc, xx -> i, yy -> i);
       }
     XFlush (display);
   } else
     px_error = PXNOWINDOWSELECTED;

  return (!px_error);
}

#else
#  ifdef PARZ

int VSetPixel (xits, yits)
ITPTRS *xits, *yits;
{ register ITEM *xit, *yit;
  
  register int *cols;

  px_error = PXOK;

  if (selectedW)
  { cols = selectedW->pixelList;
    for (pec = 0; pec < last_akt_pe; pec++, cols++)
    { xit = (* xits->n_it_func)(xits);
      yit = (* yits->n_it_func)(yits);
      if (aktive_pes[pec] == '1')
      { ini_typ_test(xit->datentyp, INT, 101, 284);
	ini_typ_test(yit->datentyp, INT, 102, 285);
	XSetForeground( display, gc, *cols);
	XDrawPoint(display, selectedW->wPointer, gc,
		   xit->inhalt.i_val, yit->inhalt.i_val);
      }
    }
    XFlush (display);
  }
  else
  { px_error = PXNOWINDOWSELECTED; }
  
  return (!px_error);
}

#  else
int VSetPixel (indx, x, y, nxproc, nyproc)

  int      indx;
  char    *x, *y;
  int      nxproc, nyproc;

{
   register int         i, j, *cols, max;
   register int        *bufX, *bufY;
            XColor      color;
   extern   int         peMax;
   extern   char       *New ();
#ifdef PZ2MPL
   extern   MemItem     _DONE;
#endif
   px_error = PXOK;

   /* Kopiere die Daten vom Back-End zum Front-End */
   bufX = (int *) New (sizeof (int) * nyproc * nxproc);
   bufY = (int *) New (sizeof (int) * nyproc * nxproc);
   blockIn (x, bufX, 0, 0, nxproc, nyproc, sizeof (INT));
   blockIn (y, bufY, 0, 0, nxproc, nyproc, sizeof (INT));

   /* Verarbeite die Daten */
   if (selectedW) {
#ifdef PZ2MPL
     _DONE.b = 1;
#endif
     cols = & (selectedW -> pixelList [indx * nxproc * nyproc]);
     for (i = 0, j = 0; i < nxproc * nyproc; i++) {
       if (bufX [i] != -99) {
         XSetForeground (display, gc, cols [j++]);
         XDrawPoint (display, selectedW -> wPointer, gc, bufX [i], bufY [i]);
       }
     }
     XFlush (display);
   } else {
#ifdef PZ2MPL
     _DONE.b = 0;
#endif
     px_error = PXNOWINDOWSELECTED;
   }

   free (bufX);
   free (bufY);
   return (!px_error);
}
#  endif
#endif


/*****************************************************************************/

int GetPixel( x, y, rgb)

  int x, y;
  COLREC *rgb;

{
  XColor       color;
  XImage       *image;

  px_error = PXOK;

  if( selectedW )
  {
/*    image = XGetImage( display, selectedW->wPointer, 0, 0, selectedW->width,
                       selectedW->height, AllPlanes, XYPixmap );

    color.pixel = XGetPixel( image, x, y );
*/
    if (x >= 0 && y >= 0 && x < selectedW->width && y < selectedW->height)
      {
	image = XGetImage( display, selectedW->wPointer, x, y, 1,
			  1, AllPlanes, XYPixmap );

	color.pixel = XGetPixel( image, 0, 0 );

	XQueryColor( display, color_map, &color );

	rgb -> r.i = (int)(color.red   / 257);
	rgb -> g.i = (int)(color.green / 257);
	rgb -> b.i = (int)(color.blue  / 257);
      }
    else
      px_error = PXWRONGCOORDINATES;
  }
  else
    px_error = PXNOWINDOWSELECTED;
  
  return (!px_error);
}


#ifdef PZ2MPLS

int GetPixelFE (x, y, rgb)

  int x, y;
  COLREC *rgb;

{
    COLREC col;
    int    ret;

    ret = GetPixel (x, y, &col);
    copyOut (&col, rgb);
    return (ret);
}

#endif


/*****************************************************************************/
/* 8 */

int MoveTo( x, y )
int x, y;

{
  px_error = PXOK;

  if (selectedW) {
    if ((x < 0) || (y < 0) || (x > selectedW->width)
        || (y > selectedW->height) )
      px_error = PXWRONGCOORDINATES;
    else {
      selectedW->x = x;
      selectedW->y = y;
    }
  }
  else
    px_error = PXNOWINDOWSELECTED;

  return (!px_error);
}

/*****************************************************************************/
/* 9 */

int LineTo( x, y )

  int x, y;

{
  px_error = PXOK;

  if( selectedW ) {
    if( (x < 0) || (y < 0) || (x > selectedW->width)
        || (y > selectedW->height) )
      px_error = PXWRONGCOORDINATES;
    else {
      XSetForeground( display, gc, selectedW->pixel );
      XDrawLine( display, selectedW->wPointer, gc, selectedW->x,      
                 selectedW->y, x, y );
      XFlush (display);

      selectedW->x = x;
      selectedW->y = y;
    }
  } else
    px_error = PXNOWINDOWSELECTED;

  return (!px_error);
}


/*****************************************************************************/
/*	DrawString: write string 'str' at position (x,y) on active window    */

int DrawString(str)

  char	*str;

{
  px_error = PXOK;

/*printf ("DrawString (%s)\n", str);*/
  
  if (selectedW) {
    if ( (selectedW -> x < 0) || (selectedW -> y < 0) || 
	(selectedW -> x > selectedW->width) || (selectedW -> y > selectedW->height) )
      px_error = PXWRONGCOORDINATES;
    else {
      XSetForeground(display, gc, selectedW->pixel);
      XDrawString(display, selectedW->wPointer, gc, 
		           selectedW->x,
		           selectedW->y, str, strlen(str));
      XFlush (display);
    }
  } else px_error = PXNOWINDOWSELECTED;

  return (!px_error);
}

/*****************************************************************************/
/*	DrawChar: write char 'c' on active window	     */

int DrawChar (c)
  char c;
{
  char	str [4];

  str [0] = c;
  str [1] = '\0';
  return (DrawString (str));
}

/*****************************************************************************/
/*	DrawInt: write integer 'i' on active window	     */

int DrawInt(i, n)
  int	i, n;
{
  char	str [80];	/* string representation of the given integer,
			   can't be exceeded.				*/

  if (n >= 80) 
    n = 79;
  sprintf (str, "%*d", n, i);
  return (DrawString (str));
}

/*****************************************************************************/
/*	DrawReal: write Real 'f' on active window	     */

int DrawReal(r, n)
  float	r;
  int   n;
{
  char str [80];	/* string representation of the given float,
			   can't be exceeded.				*/

  if (n >= 80)
    n = 79;
  sprintf (str, "%*f", n, r);
  return (DrawString (str));
}


/*****************************************************************************/
/*	DrawReal: write Real 'f'  on active window	     */

int DrawFixPt(r, n, m)
  float	r;
  int   n, m;
{
  char str [80];	/* string representation of the given float,
			   can't be exceeded.				*/

  if (n >= 80)
    n = 79;
  sprintf (str, "%*.*f", n, m, r);
  return (DrawString (str));
}


/*****************************************************************************/
/*	DrawBool: write boolean 'i' on active window	     */

int DrawBool(i)
  int	i;
{
  char	str [80];	/* string representation of the given integer,
			   can't be exceeded.				*/

  sprintf (str, "%s", i ? "TRUE" : "FALSE");
  return (DrawString (str));
}


/*****************************************************************************/

#ifdef PZ2MPLS

char *CharsToString (chars)

  MemItem *chars;

{
  char     *str;
  MemItem  *start = chars;
  int       i = 0, len = 0;

  /* Laenge bestimmen */
  while (chars -> c != '\0') {
    chars++;
    len++;
  }

  str = (char *) New (len + 1);

  /* Nun den String kopieren */
  while (start -> c != '\0') {
    str [i++] = start -> c;
    start++;
  }
  str [i] = '\0';

  return (str);
}

#endif


/*****************************************************************************/
/*	DrawStr: write parz string 's' on active window	     */

#ifdef PZ2C

int DrawStr(s)
  char 	*s;

{
  char	*str;	/* string representation of the given integer,
			   can't be exceeded.				*/
  int    i;
  extern char *CharsToString ();

  str = CharsToString (s);
  i = DrawString (str);

  free (str);
  return (i);
}

#else

/** Hier muss der String vom back-end eingelsen werden.
    dito fuer DrawStr ***/


#endif


/*****************************************************************************/
