#include "eyebot.h"
#include "math.h"

#define row 10

#define speed 100
#define angle 100
#define angularSpeed 100
#define distance 100

#define curveAngle 40

#define turnDelay 500

#define redThreshold 60
#define multiplier 1.5
#define threshold 40

#define noRedThreshold 50

#define leftThresh 2000
#define rightThresh 2000
#define centreThresh 2500

int size = QQVGA_SIZE;
int width = QQVGA_X;
int height = QQVGA_Y;

/**
 * @brief Checks the image obtained and returns the column with the greatest number of 'colour'
 * @param img the image file
 * @return the column with the largest number of red pixels or -1 if no. pixels too low
 */
int processImage(BYTE* img) {
    //allocate space for the binary image
    BYTE binary[size/3];
    
    //checking to find red pixels (red=1, not red = 0) (in RGB)
    for(int i=0; i<size; i=i+3) {
        if((img[i]>redThreshold)&&(multiplier*img[i+1]<img[i])&&(multiplier*img[i+2]<img[i])) {
            binary[i/3] = 1;
        }
        else {
            binary[i/3] = 0;
        }
    }
    
    //checking to find red pixels (HSI)
    
    
    //optional - creating an image to display for the red pixels (binary image)
    BYTE binaryImage[size];
    for(int i=0; i<size/3; i++) {
        if(binary[i] == 1) {
            binaryImage[3*i] = 255;
            binaryImage[3*i+1] = 255;
            binaryImage[3*i+2] = 255;
        }
        else {
            binaryImage[3*i] = 0;
            binaryImage[3*i+1] = 0;
            binaryImage[3*i+2] = 0;
        }
    }
    
    //printing the colour image to screen (on the left)
    LCDImageStart(0, 0, width, height);
    LCDImage(img);
    
    //printing the binary red image (on the right)
    LCDImageStart(width, 0, width, height);
    LCDImage(binaryImage);
    
    //allocating space for the 1D histogram (finding which column has most red)
    int histogram[width];
    //ensuring all zeroed out
    for(int i =0; i<width; i++) {
        histogram[i] = 0;
    }
    
    //a counter to see how many red pixels are in the image
    int redCount = 0;
    
    //filling the histogram
    for(int i =0; i<size/3; i++) {
        if(binary[i]==1) {
            redCount++;
            histogram[i%width] = histogram[i%width]+1;
        }
    }
    
    //extra checker for lower bound of red (too low to be considered)
    if(redCount<noRedThreshold) {
        //returns -1, for not enough red
         LCDSetPrintf(row+1, 0, "Not enough red on screen                     ");
        return -1;
    }
    else {
        //finding the column with the most red pixels
        int maximum = 0;
        for(int i =0; i<width; i++) {
            if(histogram[i]>histogram[maximum]) {
                maximum = i;
            }
        }
        
        //draw a vertical line on the binary image to show which column
        LCDLine(maximum+width, 0, maximum+width, height, RED);
        //printing below the images the centre column value
        LCDSetPrintf(row+1, 0, "Centre Column: %3d                             ", maximum);
        
        return maximum;
    }
}

/**
 * @brief controls the driving of the robot based on the centre column and proximity
 * @param centre the column with the highest number of red pixels
 * @return an integer representing the control state
 *  0 for at the can
 *  1 for turning right (too close to left)
 *  2 for turning left (too close to right)
 *  3 for turning left (can too far on right)
 *  4 for turning right (can too far on left)
 *  5 for going straight
 *  -1 for an error state
 */
int driveControl(int centre) {
    //obtain the PSD laser readings for distance
    int left = PSDGetRaw(1);
    int right = PSDGetRaw(3);
    int middle = PSDGetRaw(2);
    
    //Print the PSD readings
    LCDSetPrintf(row+2, 0, "PSDs: 1:%d 2:%d 3:%d                             ", left, middle, right);
    
    //hit the can and now returning
    if(middle>centreThresh) {
        LCDSetPrintf(row+3, 0, "Front too close, stopping...                 ");
        VWStraight(0,0);
        VWDriveWait();
        return 0;
    }
    
    //too close on the left
    else if(left>leftThresh) {
        LCDSetPrintf(row+3, 0, "Left side too close, turning right           ");
        VWCurve(distance, -1*curveAngle, angularSpeed);
        usleep(turnDelay);
        return 1;
    }
    //too close on the right
    else if(right>rightThresh) {
        LCDSetPrintf(row+3, 0, "Right side too close, turning left           ");
        
        VWCurve(distance,curveAngle, angularSpeed);
        usleep(turnDelay);
        return 2;
    }
    
    else {
        //shift left
        if(centre<(width/2-threshold)) {
            LCDSetPrintf(row+3, 0, "Centering image... Turning Left          ", centre);

            VWCurve(distance,curveAngle, angularSpeed);

            usleep(turnDelay);
            return 3;
        }
        //shift right
        else if (centre>(width/2+threshold)) {
            LCDSetPrintf(row+3, 0, "Centering image... Turning Right          ", centre);

            VWCurve(distance,-1*curveAngle, angularSpeed);
            usleep(turnDelay);
            return 4;
        }
        //going straight
        else {
            LCDSetPrintf(row+3, 0, "Image centred! Driving straight          ", centre);
            VWStraight(distance, speed);
            usleep(turnDelay);
            return 5;
        }
    }
    
    LCDSetPrintf(row+3, 0, "Error...                          ");
    return -1;
}


int main() {
    
    //init menu
    LCDMenu("Start", "Drive", " ", "Stop");
    
    //start key reads
    int keycode = KEYRead();
    int centre;
    int control = 9;
    
    //starting camera
    CAMInit(QQVGA);
    
    BYTE img[size];
    
    //waiting to start
    LCDSetPrintf(row, 0, "Waiting to start... Press Start                    ");
    KEYWait(KEY1);
    
    while(keycode!=KEY4) {
        //finished, restarting position
        if(control == 9) {
            LCDSetPrintf(row, 0, "Waiting for drive... Press Drive          ");
            LCDSetPrintf(row+3, 0, "                                ");
            while(keycode!=KEY2) {
                CAMGet(img);
                centre = processImage(img);
                LCDCircle(width/2, height/2, 5, ORANGE, 0);
                LCDSetPrintf(row+2, 0, "Centre pixel: R:%3d G:%3d B:%3d          ", img[3*(width*(height/2)+width/2)], img[3*((width*(height/2)+width/2))+1], img[3*((width*(height/2)+width/2))+2]);
                if(keycode == KEY4) {
                    break;
                }
                keycode = KEYRead();
            }
            //Broken out of while loop, set initial position
             LCDSetPrintf(row, 0, "Driving....                ");
            VWSetPosition(0,0,0);
            control = 10;
        }
        
        //On the return path!!!
        else if(control == 0) {
            LCDSetPrintf(row, 0, "Finished control, Returning....  ");
            
            LCDSetPrintf(row, 0, "Reversing...                     ");
            LCDSetPrintf(row+3, 0, "                                ");
            VWStraight(-50, speed);
            VWDriveWait();
            
            int x, y, phi;
            int dist;
            
            //Get the information to return
            VWGetPosition(&x, &y ,&phi);
            
            long x2 = x*x;
            long y2 = y*y;
            
            //calculate the distance to return
            dist = (int)sqrt(x2+y2);
            
            LCDSetPrintf(row+2, 0, "x: %d y: %d, phi: %d             ", x, y, phi);
            LCDSetPrintf(row+3, 0, "Distance: %d                     ", (int)dist);
            
            LCDSetPrintf(row, 0, "Turning around...               ");
            double theta = atan2(x,y);
            
            int turnAngle =270-phi-(int)((theta/M_PI)*180);
            
            VWTurn(turnAngle, angularSpeed);
            VWDriveWait();
            
            LCDSetPrintf(row, 0, "Returning...                     ");
            VWStraight((int)dist, speed);
            VWDriveWait();
            
            VWTurn(180, angularSpeed);
            VWDriveWait();
            LCDSetPrintf(row+1, 0, "Completed!                        ");

            control = 9;
        }
        
        else if (control == -1){

        }
        
        //Otherwise drive towards can
        else {
            CAMGet(img);
            centre = processImage(img);
            
            //automatically turn if theres not enough red
           if(centre == -1) {
                LCDSetPrintf(row, 0, "Not enough red on screen          ");
            }
            
            else {
                control = driveControl(centre);
            }
        }
        keycode = KEYRead();
    }
    return 0;
}
