package sim.lib.others;

import java.awt.*;

import sim.*;
import sim.engine.*;
import sim.lib.wires.Junction;
import sim.lib.wires.JunctionList;

public class Demultiplexer extends WrapperPainted implements EngineModule
{
/* ==================================================================
	Creation Part
	================================================================= */
	public Image getIcon()
	{
		return GuiFileLink.getImage("sim/lib/others/DemuxIcon.gif");
	}
	
	public Wrapper createWrapper()
	{
		return this.getCopy();
	}
	
	public Wrapper createWrapper(Point gridPosition)
	{
		Demultiplexer result = this.getCopy();
		result.setGridLocation(gridPosition);
		return result;
	}
	
	public String getBubbleHelp()
	{
		return "Demultiplexer";
	}
	
/* ==================================================================
	GUI part
	================================================================= */
	public Demultiplexer()
	{
		super();
		
		this.setOutputSize(2);
		this.setBusSize(8);
	}
	
	private Demultiplexer getCopy()
	{
		Demultiplexer result = new Demultiplexer();
		
		result.setOutputSize(this.outputSize);
		result.setBusSize(this.busSize);
		result.changeDelay(this.delay);
		
		return result;
	}
	
	public void initializeGridSize()
	{
		this.setGridSize(4, 6);
	}
	
	public void paint(Graphics g)
	{
		// draw if visible
		if(this.isVisible())
		{
			int gridGap = CentralPanel.ACTIVE_GRID.getCurrentGridGap();
			int increment = gridGap / 4;
			int middle = this.outputSize / 2;
			
			g.setColor(this.brush);
			
			int up = 2 * gridGap;
			int height = gridGap * this.getGridSize().height;
			int center = 6 * increment;
			
			g.drawRect(gridGap, center, up, height - 2 * center);
			
			if(this.requiredSelect == 1)
				g.drawLine(up, 0, up, center);
			else
				g.fillRect(up - 1, 0, 3, center);
			
			center = height / 2;
			g.setFont(new Font(Wrapper.FONT_NAME, Font.PLAIN, 3 * increment));
			
			int offset = 0;
			int thickness = 1;
			int tripple = 3 * gridGap;
			int textOffset = tripple - increment;
			FontMetrics fm = this.getFontMetrics(this.getFont());
			
			if(this.busSize != 1)
			{
				offset = -1;
				thickness = 3;
			}
			
			g.fillRect(0, center + offset, gridGap, thickness);
			
			for(height = 0; height < middle; height++)
			{
				g.drawString(Integer.toString(height), textOffset - fm.stringWidth(Integer.toString(height)), up + increment);
				g.fillRect(tripple, up + offset, gridGap, thickness);
				up = up + gridGap;
			}
			
			if((this.outputSize % 2 ) != 0)
			{
				g.drawString(Integer.toString(height), textOffset - fm.stringWidth(Integer.toString(height)), up + increment);
				g.fillRect(tripple, up + offset, gridGap, thickness);
				up = up + gridGap;
				
				for(height = middle + 1; height < this.outputSize; height++)
				{
					g.drawString(Integer.toString(height), textOffset - fm.stringWidth(Integer.toString(height)), up + increment);
					g.fillRect(tripple, up + offset, gridGap, thickness);
					up = up + gridGap;
				}
			}
			else
			{
				up = up + gridGap;
				
				for(height = middle; height < this.outputSize; height++)
				{
					g.drawString(Integer.toString(height), textOffset - fm.stringWidth(Integer.toString(height)), up + increment);
					g.fillRect(tripple, up + offset, gridGap, thickness);
					up = up + gridGap;
				}
			}
		}
	}
	
/* ==================================================================
	Maintanance Part
	================================================================= */
	private JunctionList output = new JunctionList();
	private Junction select = null;
	private Junction input = null;
	private int outputSize;
	private int requiredSelect;
	private int busSize;
	
	
	public void setOutputSize(int size)
	{
		this.outputSize = size;
				
		this.requiredSelect = (int)Math.ceil(Math.log(size) / Math.log(2));
		
		this.setGridSize(4, 2 * (size / 2) + 4);
	}
	
	public void setBusSize(int size)
	{
		this.busSize = size;
	}
	
	public boolean canDrop()
	{
		boolean result = true;
		int middle = this.outputSize / 2;
		int end = this.gridLocation.x + this.gridSize.width;
		int loop;
		
		// check outputs
		for(loop = 0; (loop < middle) && result; loop++)
			result = Wrapper.canDropJuncion(end, this.gridLocation.y + 2 + loop, this.busSize);
		
		if(result)
		{
			if((this.outputSize % 2) != 0)
			{
				result = Wrapper.canDropJuncion(end, this.gridLocation.y + 2 + middle, this.busSize);
				for(loop = middle + 1; (loop < this.outputSize) && result; loop++)
					result = Wrapper.canDropJuncion(end, this.gridLocation.y + 2 + loop, this.busSize);
			}
			else
			{
				for(loop = middle; (loop < this.outputSize) && result; loop++)
					result = Wrapper.canDropJuncion(end, this.gridLocation.y + 3 + loop, this.busSize);
			}
		}
		
		// check imputs
		if(result)
			result = Wrapper.canDropJuncion(this.gridLocation.x + 2, this.gridLocation.y, this.requiredSelect);
		
		if(result)
			result = Wrapper.canDropJuncion(this.gridLocation.x, this.gridLocation.y + 2 + middle, this.busSize);
		
		return result;
	}
		
	public void droped()
	{
		int loop;
		int middle = this.outputSize / 2;
		int end = this.gridLocation.x + this.gridSize.width;
		
		this.output.setSize(this.outputSize);
		
		for(loop = 0; loop < middle; loop++)
			this.output.changeItem(loop, Wrapper.setPinAt(end, this.gridLocation.y + 2 + loop, this.busSize));
		
		if((this.outputSize % 2) != 0)
		{
			this.output.changeItem(middle, Wrapper.setPinAt(end, this.gridLocation.y + 2 + middle, this.busSize));
			for(loop = middle + 1; loop < this.outputSize; loop++)
				this.output.changeItem(loop, Wrapper.setPinAt(end, this.gridLocation.y + 2 + loop, this.busSize));
		}
		else
		{
			for(loop = middle; loop < this.outputSize; loop++)
				this.output.changeItem(loop, Wrapper.setPinAt(end, this.gridLocation.y + 3 + loop, this.busSize));
		}
		
		this.select = Wrapper.setPinAt(this.gridLocation.x + 2, this.gridLocation.y, this.requiredSelect);
		this.input = Wrapper.setPinAt(this.gridLocation.x, this.gridLocation.y + 2 + middle, this.busSize);
		
		this.changeColor(Color.black);
		this.oldBusSize = 0;
	}
	
	public void selected()
	{
		for(int loop = 0; loop < this.outputSize; loop++)
			this.output.getItemAt(loop).removePin();
		
		this.input.removePin();
		this.select.removePin();
		
		this.changeColor(Color.green);
	}
	
	public void checkAfterSelected()
	{
		for(int loop = 0; loop < this.outputSize; loop++)
			Wrapper.checkPin(this.output.getItemAt(loop));
		
		Wrapper.checkPin(this.input);
		Wrapper.checkPin(this.select);
	}
	
/* ==================================================================
	Simulation part
	================================================================= */
	protected double delay = 1;
	
	public void evaluateOutput(double currentTime, Data[] currentInputs, EnginePeer peer)
	{
		boolean foundUndefined = false;
		int position = 0;
		int base = 1;
		int end;
		double time = currentTime + this.delay;
		
		int loop;
		
		for(loop = 0; (loop < this.requiredSelect) && (!foundUndefined); loop++)
		{
			if(currentInputs[loop].isUndefined())
			{
				foundUndefined = true;
			}
			else
			{
				if(currentInputs[loop].getValue())
					position = position + base;
				
				base = 2 * base;
			}
		}
		
		if(foundUndefined || (position >= this.outputSize))
		{
			base = this.busSize * this.outputSize;
			
			for(loop = 0; loop < base; loop++)
				peer.setOutputPinUndefined(loop, time);
		}
		else
		{
			position = position * this.busSize;
			
			for(loop = 0; loop < position; loop++)
			{
				peer.setOutputPinValue(loop, false, time);
			}
			
			base = this.requiredSelect;
			end = position + this.busSize;
			
			for(loop = position; loop < end; loop++)
			{
				if(currentInputs[base].isUndefined())
					peer.setOutputPinUndefined(loop, time);
				else
					peer.setOutputPinValue(loop, currentInputs[base].getValue(), time);
				
				base++;
			}
			
			base = this.busSize * this.outputSize;
			for(loop = end; loop < base; loop++)
			{
				peer.setOutputPinValue(loop, false, time);
			}
		}
	}
	
	public void createEnginePeer(EnginePeerList epl)
	{
		EnginePeer ep = new EnginePeer(this.busSize + this.requiredSelect, this.outputSize * this.busSize, this);
		
		int loop, index, position;
		
		for(loop = 0; loop < this.requiredSelect; loop++)
			ep.setInputPin(loop, this.select.getNodes().getItemAt(loop));
		
		position = this.requiredSelect;
		
		for(index = 0; index < this.busSize; index++)
		{
			ep.setInputPin(position, this.input.getNodes().getItemAt(index));
			position++;
		}
		
		position = 0;
		
		for(index = 0; index < this.outputSize; index++)
			for(loop = 0; loop < this.busSize; loop++)
			{
				ep.setOutputPin(position, this.output.getItemAt(index).getNodes().getItemAt(loop));
				position ++;
			}
		
		epl.insertItem(ep);
	}
	
	public void reset()
	{	
	}

	public Wrapper getParentWrapper()
	{
		return this;
	}

	public double getDelay()
	{
		return this.delay;
	}
	
	public void changeDelay(double newValue)
	{
		this.delay = newValue;
	}
	
/* ==================================================================
	Storage Part
	================================================================= */
	public String getSpecificParameters()
	{
		return (Integer.toString(this.outputSize) + Wrapper.SEPARATOR + Integer.toString(this.busSize) + Wrapper.SEPARATOR + Double.toString(this.delay) + Wrapper.SEPARATOR);
	}
	
	public void loadWrapper(String[] specificParameters) throws SimException
	{
		if(specificParameters.length == this.getNumberOfSpecificParameters())
		{
			try
			{
				this.setOutputSize(Integer.valueOf(specificParameters[0]).intValue());
				this.setBusSize(Integer.valueOf(specificParameters[1]).intValue());
				this.delay = Double.valueOf(specificParameters[2]).doubleValue();
			}
			catch(NumberFormatException e)
			{
				throw (new SimException("incorrect parameter type"));
			}
		}
		else
			throw (new SimException("incorrect number of parameters"));
	}
	
	public int getNumberOfSpecificParameters()
	{
		return 3;
	}
	
/* ==================================================================
	Popup Part
	================================================================= */
	private int oldBusSize = 0;
	private int oldSelectSize;
	
	public boolean hasProperties()
	{
		return true;
	}
	
	public Component getPropertyWindow()
	{
		return (new MuxProperties(this.busSize, this.requiredSelect, this.delay));
	}
		
	public void respondToChanges(Component property)
	{
		MuxProperties dp = (MuxProperties)property;
		
		this.delay = dp.getDelay();
		
		if(this.oldBusSize == 0)
		{
			this.oldBusSize = this.busSize;
			this.oldSelectSize = this.requiredSelect;
		}
		
		CentralPanel.ACTIVE_GRID.eraseComponent(this);
		this.setBusSize(dp.getBusSize());
		this.setOutputSize((int)Math.pow(2, dp.getInputSize()));
		CentralPanel.ACTIVE_GRID.paintComponent(this);
		
	}
	
	public void restoreOriginalProperties()
	{
		if(this.oldBusSize != 0)
		{
			this.setBusSize(this.oldBusSize);
			this.oldBusSize = 0;
			this.setOutputSize((int)Math.pow(2, this.oldSelectSize));
		}
	}
}
